#include <io.h>
#include <fcntl.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h> /* memset */
#include <math.h> /* sqrt */
#include <mx/gl.h>
#include "common.h"
//#include "anorms.h"
//#include "md2.h"


//float avertexnormals[NUMVERTEXNORMALS][3];


//static int g_glcmds = 1; /* use glcommands */
//static int g_interp = 1; /* interpolate frames */

/*
 * load model
 */
md2_model_t* md2_readModel (const char *filename)
{
	FILE *file=NULL;
	md2_model_t *model;
	byte buffer[MD2_MAX_FRAMESIZE];
	int i;
	long fLen;

	model = (md2_model_t *) malloc (sizeof (md2_model_t));
	if (!model)
		return 0;

	if (fopen_s(&file,filename, "rb"))
	{
		free (model);
		return 0;
	}

	//g_glcmds = 0; /* use glcommands */
	//g_interp = 1; /* interpolate frames */

	/* initialize model and read header */
	memset (model, 0, sizeof (md2_model_t));
	fread (&model->header, sizeof (md2_header_t), 1, file);

	fseek(file, 0L, SEEK_END);
	fLen = ftell(file);

#if 0
	printf("magic:\t\t%d\n", model->header.magic);
	printf("version:\t\t%d\n", model->header.version);
	printf("skinWidth:\t\t%d\n", model->header.skinWidth);
	printf("skinHeight:\t\t%d\n", model->header.skinHeight);
	printf("frameSize:\t\t%d\n", model->header.frameSize);
	printf("numSkins:\t\t%d\n", model->header.numSkins);
	printf("numVertices:\t\t%d\n", model->header.numVertices);
	printf("numTexCoords:\t\t%d\n", model->header.numTexCoords);
	printf("numTriangles:\t\t%d\n", model->header.numTriangles);
	printf("numGlCommands:\t\t%d\n", model->header.numGlCommands);
	printf("numFrames:\t\t%d\n", model->header.numFrames);
	printf("offsetSkins:\t\t%d\n", model->header.offsetSkins);
	printf("offsetTexCoords:\t%d\n", model->header.offsetTexCoords);
	printf("offsetTriangles:\t%d\n", model->header.offsetTriangles);
	printf("offsetFrames:\t\t%d\n", model->header.offsetFrames);
	printf("offsetGlCommands:\t%d\n", model->header.offsetGlCommands);
	printf("offsetEnd:\t\t%d\n", model->header.offsetEnd);
#endif
	if (model->header.magic != (int)(('2' << 24) + ('P' << 16) + ('D' << 8) + 'I') ||
		//do more checks
		model->header.numGlCommands < 4 ||
		fLen < model->header.offsetEnd || model->header.offsetEnd <= 0 || 
		fLen < model->header.offsetFrames || model->header.offsetFrames <= 0 ||
		fLen < model->header.offsetGlCommands || model->header.offsetGlCommands <= 0||
		fLen < model->header.offsetSkins || model->header.offsetSkins <= 0||
		fLen < model->header.offsetTexCoords ||  model->header.offsetTexCoords<= 0||
		fLen < model->header.offsetTriangles || model->header.offsetTriangles <= 0	)
	{
		fclose (file);
		free (model);
		return 0;
	}


	/* read skins */
	fseek (file, model->header.offsetSkins, SEEK_SET);
	if (model->header.numSkins > 0)
	{
		model->skins = (md2_skin_t *) malloc (sizeof (md2_skin_t) * model->header.numSkins);
		if (!model->skins)
		{
			fclose(file); //hypov8
			md2_freeModel (model);
			return 0;
		}

		fread(model->skins, sizeof(md2_skin_t), model->header.numSkins, file);
		//for (i = 0; i < model->header.numSkins; i++)
		//	fread (&model->skins[i], sizeof (md2_skin_t), 1, file);
	}

	/* read texture coordinates */
	//hypov8 todo: this is loosing uv texture precision
	fseek (file, model->header.offsetTexCoords, SEEK_SET);
	if (model->header.numTexCoords > 0)
	{
		model->texCoords = (md2_textureCoordinate_t *) malloc (sizeof (md2_textureCoordinate_t) * model->header.numTexCoords);
		if (!model->texCoords)
		{
			fclose(file); //hypov8
			md2_freeModel (model);
			return 0;
		}
		fread (model->texCoords, sizeof (md2_textureCoordinate_t), model->header.numTexCoords, file);

		//for (i = 0; i < model->header.numTexCoords; i++)
		//	fread (&model->texCoords[i], sizeof (md2_textureCoordinate_t), 1, file);
	}

	/* read triangles */
	fseek (file, model->header.offsetTriangles, SEEK_SET);
	if (model->header.numTriangles > 0)
	{
		model->triangles = (md2_triangle_t *) malloc (sizeof (md2_triangle_t) * model->header.numTriangles);
		if (!model->triangles)
		{
			fclose(file); //hypov8
			md2_freeModel (model);
			return 0;
		}
		fread (model->triangles, sizeof (md2_triangle_t), model->header.numTriangles, file);
		//for (i = 0; i < model->header.numTriangles; i++)
		//	fread (&model->triangles[i], sizeof (md2_triangle_t), 1, file);
	}

	/* read alias frames */
	fseek (file, model->header.offsetFrames, SEEK_SET);
	if (model->header.numFrames > 0)
	{
		model->frames = (md2_frame_t *) malloc (sizeof (md2_frame_t) * model->header.numFrames);
		if (!model->frames)
		{
			fclose(file); //hypov8
			md2_freeModel (model);
			return 0;
		}

		//hypov8 get bbox
		model->min[0] = model->min[1] = model->min[2] = 999999.0f;
		model->max[0] = model->max[1] = model->max[2] = -999999.0f;

		for (i = 0; i < model->header.numFrames; i++)
		{
			md2_alias_frame_t *frame = (md2_alias_frame_t *) buffer;
			int j;

			fread (frame, 1, model->header.frameSize, file);
			//strcpy_s (model->frames[i].name, sizeof(model->frames[i].name), frame->name);
			memcpy(model->frames[i].name, frame->name, sizeof(frame->name));

			model->frames[i].vertices = (md2_triangleVertex_t *) malloc (sizeof (md2_triangleVertex_t) * model->header.numVertices);
			if (!model->frames[i].vertices)
			{
				fclose(file); //hypov8
				md2_freeModel (model);
				return 0;
			}


			for (j = 0; j < model->header.numVertices; j++)
			{
				model->frames[i].vertices[j].vertex[0] = (float) ((int) frame->alias_vertices[j].vertex[0]) * frame->scale[0] + frame->translate[0];
				model->frames[i].vertices[j].vertex[2] = -1* ((float) ((int) frame->alias_vertices[j].vertex[1]) * frame->scale[1] + frame->translate[1]);
				model->frames[i].vertices[j].vertex[1] = (float) ((int) frame->alias_vertices[j].vertex[2]) * frame->scale[2] + frame->translate[2];

				model->frames[i].vertices[j].normal[0] = avertexnormals[frame->alias_vertices[j].lightNormalIndex][0];
				model->frames[i].vertices[j].normal[1] = avertexnormals[frame->alias_vertices[j].lightNormalIndex][1];
				model->frames[i].vertices[j].normal[2] = avertexnormals[frame->alias_vertices[j].lightNormalIndex][2];
				//get model bbox min/max
				if (i == 0)
				{
					int bbox;
					for (bbox = 0;bbox < 3;bbox++)
					{
						if (model->frames[i].vertices[j].vertex[bbox] < model->min[bbox])
							model->min[bbox] = model->frames[i].vertices[j].vertex[bbox];
						if (model->frames[i].vertices[j].vertex[bbox] > model->max[bbox])
							model->max[bbox] = model->frames[i].vertices[j].vertex[bbox];
					}
				}
			}
		}

		//hypov8 store frame data for later export
		fseek(file, model->header.offsetFrames, SEEK_SET);
		model->framesBuffer = (byte *)malloc(sizeof(byte) * (model->header.frameSize * model->header.numFrames));
		if (!model->framesBuffer)
		{
			fclose(file);
			md2_freeModel (model);
			return 0;
		}
		fread(model->framesBuffer, sizeof(byte), model->header.frameSize * model->header.numFrames, file);
	}

	/* read gl commands */
	fseek (file, model->header.offsetGlCommands, SEEK_SET);
	if (model->header.numGlCommands)
	{
		model->glCommandBuffer = (int *) malloc (sizeof (int) * model->header.numGlCommands);
		if (!model->glCommandBuffer)
		{
			fclose(file);
			md2_freeModel (model);
			return 0;
		}
		fread (model->glCommandBuffer, sizeof (int), model->header.numGlCommands, file);
	}

	fclose (file);

	return model;
}

mdx_model_t * md2_Parse_readModel(const char * filename)
{
	int i;
	md2_model_t *md2Src;
	mdx_model_t *mdxOut;

	md2Src = md2_readModel(filename);
	if (!md2Src)
		return 0;

	//convert md2 to mdx viewer usable format
	mdxOut = (mdx_model_t *)malloc(sizeof(mdx_model_t));
	if (!mdxOut)
		return 0;


	//header
	memset(mdxOut, 0, sizeof(mdx_model_t));
	memcpy(&mdxOut->header, &md2Src->header, sizeof(int) * 7);
	mdxOut->header.numTriangles = md2Src->header.numTriangles;
	mdxOut->header.numGlCommands = md2Src->header.numGlCommands;
	mdxOut->header.numFrames = md2Src->header.numFrames;

	memcpy(mdxOut->min, md2Src->min, sizeof(float) * 3);
	memcpy(mdxOut->max, md2Src->max, sizeof(float) * 3);

	//skins
	mdxOut->skins = (mdx_skin_t *)malloc(sizeof(md2_skin_t)*md2Src->header.numSkins);
	if (mdxOut->skins)
	{
		memcpy(mdxOut->skins, md2Src->skins, sizeof(md2_skin_t)*md2Src->header.numSkins);

		//tex cords

		//triangles
		mdxOut->triangles = (mdx_triangle_t *)malloc(sizeof(md2_triangle_t) * md2Src->header.numTriangles);
		if (mdxOut->triangles)
		{
			memcpy(mdxOut->triangles, md2Src->triangles, sizeof(md2_triangle_t)*md2Src->header.numTriangles);

			//frames
			mdxOut->frames = (mdx_frame_t *)malloc(sizeof(md2_frame_t) * md2Src->header.numFrames);
			if (mdxOut->frames)
			{
				for (i = 0;i < md2Src->header.numFrames; i++)
				{
					//memcpy(mdxOut->frames[i].scale, md2Src->frames[i].scale, sizeof(float) * 3);
					//memcpy(mdxOut->frames[i].translate, md2Src->frames[i].translate, sizeof(float) * 3);
					memcpy(mdxOut->frames[i].name, md2Src->frames[i].name, sizeof(md2Src->frames[i].name));
					
					//vertex	
					mdxOut->frames[i].vertices = (mdx_triangleVertex_t *)malloc(sizeof(mdx_triangleVertex_t) * md2Src->header.numVertices);
					if (!mdxOut->frames[i].vertices)
					{
						mdx_freeModel(mdxOut);
						md2_freeModel(md2Src);
						return 0;
					}
					memcpy(mdxOut->frames[i].vertices, md2Src->frames[i].vertices, sizeof(mdx_triangleVertex_t)*md2Src->header.numVertices);
				}

				//frameBuffer
				mdxOut->framesBuffer = (byte *)malloc(sizeof(byte) * (md2Src->header.frameSize * md2Src->header.numFrames));
				if (mdxOut->framesBuffer)
				{
					memcpy(mdxOut->framesBuffer, md2Src->framesBuffer, sizeof(byte) * (md2Src->header.frameSize * md2Src->header.numFrames));
					
					//glCommand buffer
					mdxOut->glCommandBuffer = (int *)malloc(sizeof(int) * md2Src->header.numGlCommands);
					if (mdxOut->glCommandBuffer)
					{
						memcpy(mdxOut->glCommandBuffer, md2Src->glCommandBuffer, sizeof(int) * md2Src->header.numGlCommands);

						mdxOut->isMD2 = 1; //stop reading object index in glCommands
						//setModelIndex(); //hypov8

						md2_freeModel(md2Src);

						return mdxOut;
					}
				}
			}
		}
	}


	//must have failed!!
	mdx_freeModel(mdxOut);
	md2_freeModel(md2Src);
	return 0;
}



/*
 * free model
 */
void
md2_freeModel (md2_model_t *model)
{
	if (model)
	{
		if (model->skins)
			free (model->skins);

		if (model->texCoords)
			free (model->texCoords);

		if (model->triangles)
			free (model->triangles);

		if (model->frames)
		{
			int i;

			for (i = 0; i < model->header.numFrames; i++)
			{
				if (model->frames[i].vertices)
					free(model->frames[i].vertices);
			}
			free(model->frames);
		}

		if (model->framesBuffer)
			free(model->framesBuffer);

		if (model->glCommandBuffer)
			free (model->glCommandBuffer);


		free (model);
	}
}


#if 1 //md2viewer
/*
* set draw style, either with glcommands, interpolated
*/
/*void
md2_setStyle(int glcmds, int interp)
{
	g_glcmds = glcmds;
	g_interp = interp;
}*/



/*
* center model
*
*/
void
md2_getBoundingBox(md2_model_t *model, float *minmax)
{
	int i;
	float minx, maxx;
	float miny, maxy;
	float minz, maxz;

	minx = miny = minz = 999999.0f;
	maxx = maxy = maxz = -999999.0f;

	/* get bounding box */
	for (i = 0; i < model->header.numVertices; i++)
	{
		md2_triangleVertex_t *v = &model->frames[0].vertices[i];

		if (v->vertex[0] < minx)
			minx = v->vertex[0];
		else if (v->vertex[0] > maxx)
			maxx = v->vertex[0];

		if (v->vertex[1] < miny)
			miny = v->vertex[1];
		else if (v->vertex[1] > maxy)
			maxy = v->vertex[1];

		if (v->vertex[2] < minz)
			minz = v->vertex[2];
		else if (v->vertex[2] > maxz)
			maxz = v->vertex[2];
	}

	minmax[0] = minx;
	minmax[1] = maxx;
	minmax[2] = miny;
	minmax[3] = maxy;
	minmax[4] = minz;
	minmax[5] = maxz;
}



/*
* draw normal
*
*/
void _md2_drawModel(md2_model_t *model, int frame)
{
	int i;

	md2_frame_t *f = &model->frames[frame];

	glBegin(GL_TRIANGLES);

	for (i = 0; i < model->header.numTriangles; i++)
	{
		md2_triangle_t *t = &model->triangles[i];
		//int lightNormalIndex;
		/*
		lightNormalIndex = f->vertices[t->vertexIndices[0]].lightNormalIndex;
		glNormal3f (avertexnormals[lightNormalIndex][0],
		avertexnormals[lightNormalIndex][2],
		-avertexnormals[lightNormalIndex][1]);
		*/
		glNormal3f(f->vertices[t->vertexIndices[0]].normal[0],
			f->vertices[t->vertexIndices[0]].normal[2],
			-f->vertices[t->vertexIndices[0]].normal[1]);
		glTexCoord2f((float)model->texCoords[t->textureIndices[0]].s / (float)model->header.skinWidth,
			(float)model->texCoords[t->textureIndices[0]].t / (float)model->header.skinHeight);
		glVertex3f(f->vertices[t->vertexIndices[0]].vertex[0],
			f->vertices[t->vertexIndices[0]].vertex[1],
			f->vertices[t->vertexIndices[0]].vertex[2]);

		/*
		lightNormalIndex = f->vertices[t->vertexIndices[1]].lightNormalIndex;
		glNormal3f (avertexnormals[lightNormalIndex][0],
		avertexnormals[lightNormalIndex][2],
		-avertexnormals[lightNormalIndex][1]);
		*/
		glNormal3f(f->vertices[t->vertexIndices[1]].normal[0],
			f->vertices[t->vertexIndices[1]].normal[2],
			-f->vertices[t->vertexIndices[1]].normal[1]);
		glTexCoord2f((float)model->texCoords[t->textureIndices[1]].s / (float)model->header.skinWidth,
			(float)model->texCoords[t->textureIndices[1]].t / (float)model->header.skinHeight);
		glVertex3f(f->vertices[t->vertexIndices[1]].vertex[0],
			f->vertices[t->vertexIndices[1]].vertex[1],
			f->vertices[t->vertexIndices[1]].vertex[2]);
		/*
		lightNormalIndex = f->vertices[t->vertexIndices[2]].lightNormalIndex;
		glNormal3f (avertexnormals[lightNormalIndex][0],
		avertexnormals[lightNormalIndex][2],
		-avertexnormals[lightNormalIndex][1]);
		*/
		glNormal3f(f->vertices[t->vertexIndices[2]].normal[0],
			f->vertices[t->vertexIndices[2]].normal[2],
			-f->vertices[t->vertexIndices[2]].normal[1]);
		glTexCoord2f((float)model->texCoords[t->textureIndices[2]].s / (float)model->header.skinWidth,
			(float)model->texCoords[t->textureIndices[2]].t / (float)model->header.skinHeight);
		glVertex3f(f->vertices[t->vertexIndices[2]].vertex[0],
			f->vertices[t->vertexIndices[2]].vertex[1],
			f->vertices[t->vertexIndices[2]].vertex[2]);
	}
	glEnd();
}



/*
* draw interpolated
*
*/
void _md2_drawModeli(md2_model_t *model, int frame1, int frame2, float pol)
{
	int i;

	md2_frame_t *f1 = &model->frames[frame1];
	md2_frame_t *f2 = &model->frames[frame2];

	glBegin(GL_TRIANGLES);

	for (i = 0; i < model->header.numTriangles; i++)
	{
		md2_triangle_t *t = &model->triangles[i];
		//int lightNormalIndex1, lightNormalIndex2;
		float x1, y1, z1, x2, y2, z2;
		float *n1, *n2;

		glTexCoord2f((float)model->texCoords[t->textureIndices[0]].s / (float)model->header.skinWidth,
			(float)model->texCoords[t->textureIndices[0]].t / (float)model->header.skinHeight);
		/*
		lightNormalIndex1 = f1->vertices[t->vertexIndices[0]].lightNormalIndex;
		lightNormalIndex2 = f2->vertices[t->vertexIndices[0]].lightNormalIndex;
		glNormal3f ((1.0f - pol) * avertexnormals[lightNormalIndex1][0] + pol * avertexnormals[lightNormalIndex2][0],
		(1.0f - pol) * avertexnormals[lightNormalIndex1][2] + pol * avertexnormals[lightNormalIndex2][2],
		(1.0f - pol) * -avertexnormals[lightNormalIndex1][1] + pol * -avertexnormals[lightNormalIndex2][1]);
		*/
		n1 = f1->vertices[t->vertexIndices[0]].normal;
		n2 = f2->vertices[t->vertexIndices[0]].normal;
		glNormal3f((1.0f - pol) * n1[0] + pol * n2[0],
			(1.0f - pol) * n1[2] + pol * n2[2],
			(1.0f - pol) * -n1[1] + pol * -n2[1]);

		x1 = f1->vertices[t->vertexIndices[0]].vertex[0];
		y1 = f1->vertices[t->vertexIndices[0]].vertex[1];
		z1 = f1->vertices[t->vertexIndices[0]].vertex[2];
		x2 = f2->vertices[t->vertexIndices[0]].vertex[0];
		y2 = f2->vertices[t->vertexIndices[0]].vertex[1];
		z2 = f2->vertices[t->vertexIndices[0]].vertex[2];
		glVertex3f(x1 + pol * (x2 - x1),
			y1 + pol * (y2 - y1),
			z1 + pol * (z2 - z1));

		glTexCoord2f((float)model->texCoords[t->textureIndices[1]].s / (float)model->header.skinWidth,
			(float)model->texCoords[t->textureIndices[1]].t / (float)model->header.skinHeight);
		/*
		lightNormalIndex1 = f1->vertices[t->vertexIndices[1]].lightNormalIndex;
		lightNormalIndex2 = f2->vertices[t->vertexIndices[1]].lightNormalIndex;
		glNormal3f ((1.0f - pol) * avertexnormals[lightNormalIndex1][0] + pol * avertexnormals[lightNormalIndex2][0],
		(1.0f - pol) * avertexnormals[lightNormalIndex1][2] + pol * avertexnormals[lightNormalIndex2][2],
		(1.0f - pol) * -avertexnormals[lightNormalIndex1][1] + pol * -avertexnormals[lightNormalIndex2][1]);
		*/
		n1 = f1->vertices[t->vertexIndices[1]].normal;
		n2 = f2->vertices[t->vertexIndices[1]].normal;
		glNormal3f((1.0f - pol) * n1[0] + pol * n2[0],
			(1.0f - pol) * n1[2] + pol * n2[2],
			(1.0f - pol) * -n1[1] + pol * -n2[1]);

		x1 = f1->vertices[t->vertexIndices[1]].vertex[0];
		y1 = f1->vertices[t->vertexIndices[1]].vertex[1];
		z1 = f1->vertices[t->vertexIndices[1]].vertex[2];
		x2 = f2->vertices[t->vertexIndices[1]].vertex[0];
		y2 = f2->vertices[t->vertexIndices[1]].vertex[1];
		z2 = f2->vertices[t->vertexIndices[1]].vertex[2];
		glVertex3f(x1 + pol * (x2 - x1),
			y1 + pol * (y2 - y1),
			z1 + pol * (z2 - z1));

		glTexCoord2f((float)model->texCoords[t->textureIndices[2]].s / (float)model->header.skinWidth,
			(float)model->texCoords[t->textureIndices[2]].t / (float)model->header.skinHeight);
		/*
		lightNormalIndex1 = f1->vertices[t->vertexIndices[2]].lightNormalIndex;
		lightNormalIndex2 = f2->vertices[t->vertexIndices[2]].lightNormalIndex;
		glNormal3f ((1.0f - pol) * avertexnormals[lightNormalIndex1][0] + pol * avertexnormals[lightNormalIndex2][0],
		(1.0f - pol) * avertexnormals[lightNormalIndex1][2] + pol * avertexnormals[lightNormalIndex2][2],
		(1.0f - pol) * -avertexnormals[lightNormalIndex1][1] + pol * -avertexnormals[lightNormalIndex2][1]);
		*/
		n1 = f1->vertices[t->vertexIndices[2]].normal;
		n2 = f2->vertices[t->vertexIndices[2]].normal;
		glNormal3f((1.0f - pol) * n1[0] + pol * n2[0],
			(1.0f - pol) * n1[2] + pol * n2[2],
			(1.0f - pol) * -n1[1] + pol * -n2[1]);

		x1 = f1->vertices[t->vertexIndices[2]].vertex[0];
		y1 = f1->vertices[t->vertexIndices[2]].vertex[1];
		z1 = f1->vertices[t->vertexIndices[2]].vertex[2];
		x2 = f2->vertices[t->vertexIndices[2]].vertex[0];
		y2 = f2->vertices[t->vertexIndices[2]].vertex[1];
		z2 = f2->vertices[t->vertexIndices[2]].vertex[2];
		glVertex3f(x1 + pol * (x2 - x1),
			y1 + pol * (y2 - y1),
			z1 + pol * (z2 - z1));
	}
	glEnd();
}



/*
* draw with glcommands
*
*/
#if 0
void _md2_drawModelg(md2_model_t *model, int frame)
{
	int i = 0;
	int val = model->glCommandBuffer[i++];

	while (val != 0)
	{
		int count;

		if (val > 0)
		{
			glBegin(GL_TRIANGLE_STRIP);
			count = val;
		}
		else
		{
			glBegin(GL_TRIANGLE_FAN);
			count = -val;
		}

		while (count--)
		{
			float s = *(float *)&model->glCommandBuffer[i++];
			float t = *(float *)&model->glCommandBuffer[i++];
			int index = model->glCommandBuffer[i++];
			//int lightNormalIndex = model->frames[frame].vertices[index].lightNormalIndex;

			glTexCoord2f(s, t);
			/*
			glNormal3f (avertexnormals[lightNormalIndex][0],
			avertexnormals[lightNormalIndex][2],
			-avertexnormals[lightNormalIndex][1]);
			*/
			glNormal3f(model->frames[frame].vertices[index].normal[0],
				model->frames[frame].vertices[index].normal[2],
				-model->frames[frame].vertices[index].normal[1]);
			glVertex3f(model->frames[frame].vertices[index].vertex[0],
				model->frames[frame].vertices[index].vertex[1],
				model->frames[frame].vertices[index].vertex[2]);
		}

		glEnd();

		val = model->glCommandBuffer[i++];
	}
}



/*
* draw with glcommands, interpolated
*
*/
void _md2_drawModelgi(md2_model_t *model, int frame1, int frame2, float pol)
{
	int i = 0;
	int val = model->glCommandBuffer[i++];

	while (val != 0)
	{
		int count;

		if (val > 0)
		{
			glBegin(GL_TRIANGLE_STRIP);
			count = val;
		}
		else
		{
			glBegin(GL_TRIANGLE_FAN);
			count = -val;
		}

		while (count--)
		{
			float s = *(float *)&model->glCommandBuffer[i++];
			float t = *(float *)&model->glCommandBuffer[i++];
			int index = model->glCommandBuffer[i++];
			//int lightNormalIndex1 = model->frames[frame1].vertices[index].lightNormalIndex;
			//int lightNormalIndex2 = model->frames[frame2].vertices[index].lightNormalIndex;
			float x1, y1, z1, x2, y2, z2;
			float *n1 = model->frames[frame1].vertices[index].normal;
			float *n2 = model->frames[frame2].vertices[index].normal;

			/* texcoords don't need to be interpolated */
			glTexCoord2f(s, t);

			/* interpolate light normal */
			/*
			glNormal3f ((1.0f - pol) * avertexnormals[lightNormalIndex1][0] + pol * avertexnormals[lightNormalIndex2][0],
			(1.0f - pol) * avertexnormals[lightNormalIndex1][2] + pol * avertexnormals[lightNormalIndex2][2],
			(1.0f - pol) * -avertexnormals[lightNormalIndex1][1] + pol * -avertexnormals[lightNormalIndex2][1]);
			*/
			glNormal3f((1.0f - pol) * n1[0] + pol * n2[0],
				(1.0f - pol) * n1[2] + pol * n2[2],
				(1.0f - pol) * -n1[1] + pol * -n2[1]);

			/* interpolate vertices */
			x1 = model->frames[frame1].vertices[index].vertex[0];
			y1 = model->frames[frame1].vertices[index].vertex[1];
			z1 = model->frames[frame1].vertices[index].vertex[2];

			x2 = model->frames[frame2].vertices[index].vertex[0];
			y2 = model->frames[frame2].vertices[index].vertex[1];
			z2 = model->frames[frame2].vertices[index].vertex[2];

			glVertex3f(x1 + pol * (x2 - x1),
				y1 + pol * (y2 - y1),
				z1 + pol * (z2 - z1));
		}

		glEnd();

		val = model->glCommandBuffer[i++];
	}
}



/*
* draw model
*/

void md2_drawModel(md2_model_t *model, int frame1, int frame2, float pol)
{
	if (g_glcmds)
	{
		if (g_interp)
			_md2_drawModelgi(model, frame1, frame2, pol);
		else
			_md2_drawModelg(model, frame1);
	}
	else
	{
		if (g_interp)
			_md2_drawModeli(model, frame1, frame2, pol);
		else
			_md2_drawModel(model, frame1);
	}
}
#endif


void md2_makeFacetNormal(md2_model_t *model, md2_triangle_t *t, int frame, float *fn)
{
	float v1[3], v2[3];
	double angle;

	v1[0] = model->frames[frame].vertices[t->vertexIndices[1]].vertex[0] - model->frames[frame].vertices[t->vertexIndices[0]].vertex[0];
	v1[1] = model->frames[frame].vertices[t->vertexIndices[1]].vertex[1] - model->frames[frame].vertices[t->vertexIndices[0]].vertex[1];
	v1[2] = model->frames[frame].vertices[t->vertexIndices[1]].vertex[2] - model->frames[frame].vertices[t->vertexIndices[0]].vertex[2];

	v2[0] = model->frames[frame].vertices[t->vertexIndices[2]].vertex[0] - model->frames[frame].vertices[t->vertexIndices[0]].vertex[0];
	v2[1] = model->frames[frame].vertices[t->vertexIndices[2]].vertex[1] - model->frames[frame].vertices[t->vertexIndices[0]].vertex[1];
	v2[2] = model->frames[frame].vertices[t->vertexIndices[2]].vertex[2] - model->frames[frame].vertices[t->vertexIndices[0]].vertex[2];

	angle = 1;

	fn[0] = (v1[1] * v2[2] - v1[2] * v2[1]) * (float)angle;
	fn[1] = (v1[2] * v2[0] - v1[0] * v2[2]) * (float)angle;
	fn[2] = (v1[0] * v2[1] - v1[1] * v2[0]) * (float)angle;
}



void md2_normalize(float *n)
{
	float l = (float)sqrt(n[0] * n[0] + n[1] * n[1] + n[2] * n[2]);

	if (l != 0.0f)
	{
		n[0] /= l;
		n[1] /= l;
		n[2] /= l;
	}
}



void
md2_generateLightNormals(md2_model_t *model)
{
	int i;

	if (!model)
		return;


	for (i = 0; i < model->header.numFrames; i++)
	{
		int j;

		/* clear all normals */
		for (j = 0; j < model->header.numVertices; j++)
		{
			model->frames[i].vertices[j].normal[0] = 0.0f;
			model->frames[i].vertices[j].normal[1] = 0.0f;
			model->frames[i].vertices[j].normal[2] = 0.0f;
		}

		/* calc normals */
		for (j = 0; j < model->header.numTriangles; j++)
		{
			int k;
			float facetnormal[3];

			md2_makeFacetNormal(model, &model->triangles[j], i, facetnormal);

			for (k = 0; k < 3; k++)
			{
				model->frames[i].vertices[model->triangles[j].vertexIndices[k]].normal[0] -= facetnormal[0];
				model->frames[i].vertices[model->triangles[j].vertexIndices[k]].normal[1] += facetnormal[2];
				model->frames[i].vertices[model->triangles[j].vertexIndices[k]].normal[2] -= facetnormal[1];
			}
		}

		/* normalize normals */
		for (j = 0; j < model->header.numVertices; j++)
			md2_normalize(model->frames[i].vertices[j].normal);
	}
}



int
md2_getAnimationCount(md2_model_t *model)
{
	int i, j, pos;
	int count;
	int lastId;
	char name[32], last[32];

	//strcpy_s(last, sizeof(last), model->frames[0].name);
	memcpy(last, model->frames[0].name, sizeof(model->frames[0].name));
	pos = strlen(last) - 1;
	j = 0;
	while (last[pos] >= '0' && last[pos] <= '9' && j < 2)
	{
		pos--;
		j++;
	}
	last[pos + 1] = '\0';

	lastId = 0;
	count = 0;

	for (i = 0; i <= model->header.numFrames; i++)
	{
		if (i == model->header.numFrames)
			name[0]='\0'; // some kind of a sentinel
		else
		{
			//strcpy_s (name, sizeof(model->frames[i].name), model->frames[i].name);
			memcpy(name, model->frames[i].name, sizeof(model->frames[i].name));
			name[16] = '\0'; //str not allways null terminated
		}
		pos = strlen(name) - 1;
		j = 0;
		while (name[pos] >= '0' && name[pos] <= '9' && j < 2)
		{
			pos--;
			j++;
		}
		name[pos + 1] = '\0';

		if (strcmp(last, name))
		{
			strcpy_s(last, sizeof(last), name);
			count++;
		}
	}

	return count;
}



const char *
md2_getAnimationName(md2_model_t *model, int animation)
{
	int i, j, pos;
	int count;
	int lastId;
	static char last[32];
	char name[32];

	//strcpy_s(last, sizeof(last), model->frames[0].name);
	memcpy(last, model->frames[0].name, sizeof(model->frames[0].name));
	pos = strlen(last) - 1;
	j = 0;
	while (last[pos] >= '0' && last[pos] <= '9' && j < 2)
	{
		pos--;
		j++;
	}
	last[pos + 1] = '\0';

	lastId = 0;
	count = 0;

	for (i = 0; i <= model->header.numFrames; i++)
	{
		if (i == model->header.numFrames)
			name[0] = '\0'; // some kind of a sentinel
		else
		{
			//strcpy_s (name, sizeof(model->frames[i].name), model->frames[i].name);
			memcpy(name, model->frames[i].name, sizeof(model->frames[i].name));
			name[16] = '\0'; //str not allways null terminated
		}
		pos = strlen(name) - 1;
		j = 0;
		while (name[pos] >= '0' && name[pos] <= '9' && j < 2)
		{
			pos--;
			j++;
		}
		name[pos + 1] = '\0';

		if (strcmp(last, name))
		{
			if (count == animation)
				return last;

			strcpy_s(last, sizeof(last), name);
			count++;
		}
	}

	return 0;
}



void
md2_getAnimationFrames(md2_model_t *model, int animation, int *startFrame, int *endFrame)
{
	int i, j, pos;
	int count, numFrames, frameCount;
	int lastId;
	char name[32], last[32];

	//strcpy_s(last, sizeof(last), model->frames[0].name);
	memcpy(last, model->frames[0].name, sizeof(model->frames[0].name));

	pos = strlen(last) - 1;
	j = 0;
	while (last[pos] >= '0' && last[pos] <= '9' && j < 2)
	{
		pos--;
		j++;
	}
	last[pos + 1] = '\0';

	lastId = 0;
	count = 0;
	numFrames = 0;
	frameCount = 0;

	for (i = 0; i <= model->header.numFrames; i++)
	{
		if (i == model->header.numFrames)
			name[0] = '\0'; // some kind of a sentinel
		else
		{
			//strcpy_s (name, sizeof(model->frames[i].name), model->frames[i].name);
			memcpy(name, model->frames[i].name, sizeof(model->frames[i].name));
			name[16] = '\0'; //str not allways null terminated
		}
		pos = strlen(name) - 1;
		j = 0;
		while (name[pos] >= '0' && name[pos] <= '9' && j < 2)
		{
			pos--;
			j++;
		}
		name[pos + 1] = '\0';

		if (strcmp(last, name))
		{
			strcpy_s(last, sizeof(last), name);

			if (count == animation)
			{
				*startFrame = frameCount - numFrames;
				*endFrame = frameCount - 1;
				return;
			}

			count++;
			numFrames = 0;
		}
		frameCount++;
		numFrames++;
	}


	*startFrame = *endFrame = 0;
}



#if 0
void
md2_printModelInfo(md2_model_t *model)
{
	int i;

	printf("magic:\t\t%d\n", model->header.magic);
	printf("version:\t\t%d\n", model->header.version);
	printf("skinWidth:\t\t%d\n", model->header.skinWidth);
	printf("skinHeight:\t\t%d\n", model->header.skinHeight);
	printf("frameSize:\t\t%d\n", model->header.frameSize);
	printf("numSkins:\t\t%d\n", model->header.numSkins);
	printf("numVertices:\t\t%d\n", model->header.numVertices);
	printf("numTexCoords:\t\t%d\n", model->header.numTexCoords);
	printf("numTriangles:\t\t%d\n", model->header.numTriangles);
	printf("numGlCommands:\t\t%d\n", model->header.numGlCommands);
	printf("numFrames:\t\t%d\n", model->header.numFrames);
	printf("offsetSkins:\t\t%d\n", model->header.offsetSkins);
	printf("offsetTexCoords:\t%d\n", model->header.offsetTexCoords);
	printf("offsetTriangles:\t%d\n", model->header.offsetTriangles);
	printf("offsetFrames:\t\t%d\n", model->header.offsetFrames);
	printf("offsetGlCommands:\t%d\n", model->header.offsetGlCommands);
	printf("offsetEnd:\t\t%d\n", model->header.offsetEnd);

	for (i = 0; i < model->header.numFrames; i++)
		printf("%s ", model->frames[i].name);

}

int
main(int argc, char *argv[])
{
	md2_model_t *model;

	if (argc != 2)
	{
		printf("usage: %s <model.md2>\n", argv[0]);
		return -1;
	}

	model = md2_readModel(argv[1]);
	if (!model)
	{
		printf("hoops\n");
		return -1;
	}

	md2_printModelInfo(model);
	md2_freeModel(model);

	return 0;
}
#endif





#endif