/*  Misfit Model 3D
 * 
 *  Copyright (c) 2004-2005 Kevin Worcester
 * 
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, 
 *  USA.
 *
 *  See the COPYING file for full license text.
 */


#include "rotatetool.h"

#include "model.h"
#include "pixmap/rotatetool.xpm"
#include "glmath.h"
#include "decalmgr.h"
#include "rotatepoint.h"
#include "log.h"
#include "modelstatus.h"

#include <math.h>

RotateTool::RotateTool()
   : m_tracking( false )
{
}

RotateTool::~RotateTool()
{
}

void RotateTool::activated( int arg, Model * model, QMainWindow * mainwin )
{
   model_status( model, StatusNormal, STATUSTIME_NONE, "Tip: Hold shift to rotate in 15 degree increments" );

   m_coords[0] = 0;
   m_coords[1] = 0;
   m_coords[2] = 0;

   if ( model->getAnimationMode() == Model::ANIMMODE_SKELETAL )
   {
      list<int> joints = model->getSelectedBoneJoints();
      if ( joints.size() > 0 )
      {
         model->getBoneJointCoords( joints.front(), m_coords );
      }
   }

   m_rotatePoint = new RotatePoint();
   m_rotatePoint->setPoint( m_coords[0], m_coords[1], m_coords[2] );
   DecalManager::getInstance()->addDecalToModel( m_rotatePoint, model );
}

void RotateTool::deactivated()
{
   DecalManager::getInstance()->removeDecal( m_rotatePoint );
   delete m_rotatePoint;
   m_rotatePoint = NULL;
}

void RotateTool::mouseButtonDown( Parent * parent, int buttonState, int x, int y )
{
   Model * model = parent->getModel();

   if ( model->getAnimationMode() == Model::ANIMMODE_SKELETAL )
   {
      list<int> joints = model->getSelectedBoneJoints();
      if ( joints.size() > 0 )
      {
         model->getBoneJointCoords( joints.front(), m_coords );
         m_rotatePoint->setPoint( m_coords[0], m_coords[1], m_coords[2] );
      }
   }

   double newCoords[3] = {0,0,0};

   int index = 0;

   if ( !parent->getXValue( x, y, &newCoords[0] ) )
   {
      index = 1;
   }
   if ( !parent->getYValue( x, y, &newCoords[1] ) )
   {
      index = 2;
   }
   if ( !parent->getZValue( x, y, &newCoords[2] ) )
   {
      index = 0;
   }

   if ( buttonState & BS_Left )
   {
      m_tracking = true;

      getRotateCoords( model );

      double xDiff = newCoords[0] - m_coords[0];
      double yDiff = newCoords[1] - m_coords[1];
      double zDiff = newCoords[2] - m_coords[2];

      double angle = 0;

      switch ( index )
      {
         case 0:
            angle = diffToAngle( yDiff, xDiff );
            break;
         case 1:
            angle = diffToAngle( yDiff, zDiff );
            break;
         case 2:
            angle = diffToAngle( zDiff, xDiff );
            break;
         default:
            break;
      }

      if ( buttonState & BS_Shift )
      {
         angle = adjustToNearest( angle );
      }
      m_startAngle = angle;
      model_status( model, StatusNormal, STATUSTIME_SHORT, "Rotating selected primitives" );
   }
   else if ( buttonState & BS_Right && model->getAnimationMode() != Model::ANIMMODE_SKELETAL )
   {
      for ( int t = 0; t < 3; t++ )
      {
         m_coords[t] = newCoords[t];
      }

      m_rotatePoint->setPoint( m_coords[0], m_coords[1], m_coords[2] );
      model_status( model, StatusNormal, STATUSTIME_SHORT, "Setting rotation point" );
   }
   parent->updateAllViews();
}

void RotateTool::mouseButtonMove( Parent * parent, int buttonState, int x, int y )
{
   Model * model = parent->getModel();

   double newCoords[3] = {0,0,0};

   int index = 0;

   if ( !parent->getXValue( x, y, &newCoords[0] ) )
   {
      index = 1;
   }
   if ( !parent->getYValue( x, y, &newCoords[1] ) )
   {
      index = 2;
   }
   if ( !parent->getZValue( x, y, &newCoords[2] ) )
   {
      index = 0;
   }

   if ( buttonState & BS_Left )
   {
      getRotateCoords( model );

      double xDiff = newCoords[0] - m_coords[0];
      double yDiff = newCoords[1] - m_coords[1];
      double zDiff = newCoords[2] - m_coords[2];

      double angle = 0;
      double angles[3];
      for( int t = 0; t< 3; t++ )
      {
         angles[t] = 0;
      }
      Matrix m;

      switch ( index )
      {
         case 0:
            angle = diffToAngle( yDiff, xDiff );
            if ( buttonState & BS_Shift )
            {
               angle = adjustToNearest( angle );
            }
            angles[2] = (angle - m_startAngle);
            break;
         case 1:
            angle = diffToAngle( yDiff, zDiff );
            if ( buttonState & BS_Shift )
            {
               angle = adjustToNearest( angle );
            }
            angles[0] = -(angle - m_startAngle);
            break;
         case 2:
            angle = diffToAngle( zDiff, xDiff );
            if ( buttonState & BS_Shift )
            {
               angle = adjustToNearest( angle );
            }
            angles[1] = -(angle - m_startAngle);
            break;
         default:
            break;
      }

      m.setRotation( angles );

      model->rotateSelectedVerticesOnPoint( m, m_coords );

      m_startAngle = angle;
   }
   else if ( buttonState & BS_Right )
   {
      getRotateCoords( model );

      for ( int t = 0; t < 3; t++ )
      {
         m_coords[t] = newCoords[t];
      }

      m_rotatePoint->setPoint( m_coords[0], m_coords[1], m_coords[2] );
   }
   parent->updateAllViews();
}

void RotateTool::mouseButtonUp( Parent * parent, int buttonState, int x, int y )
{
   m_tracking = false;

   if ( buttonState & BS_Left )
   {
      model_status( parent->getModel(), StatusNormal, STATUSTIME_SHORT, "Rotate complete" );
   }
}

const char ** RotateTool::getPixmap()
{
   return (const char **) rotatetool_xpm;
}

double RotateTool::diffToAngle( double opposite, double adjacent )
{
   if ( adjacent < 0.0001 && adjacent > -0.0001 )
   {
      adjacent = (adjacent >= 0 ) ? 0.0001 : -0.0001;
   }

   double angle = atan(  opposite / adjacent );

   float quad = PIOVER180 * 90;

   if ( adjacent < 0 )
   {
      if ( opposite < 0 )
      {
         angle = -(quad) - ( (quad) - angle );
      }
      else
      {
         angle = (quad) + ( (quad) + angle );
      }
   }

   return angle;
}

void RotateTool::getRotateCoords( Model * model )
{
   m_rotatePoint->getPoint( m_coords[0], m_coords[1], m_coords[2] );
}

double RotateTool::adjustToNearest( double angle )
{
   double f = angle / PIOVER180; // Change to degrees
   if ( f < 0.0 )
   {
      int n = (int) (f / 15.0 - 0.5);
      f = n * 15.0;
   }
   else
   {
      int n = (int) (f / 15.0 + 0.5);
      f = n * 15.0;
   }
   return f * PIOVER180;
}

