/*  Misfit Model 3D
 * 
 *  Copyright (c) 2004-2005 Kevin Worcester
 * 
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, 
 *  USA.
 *
 *  See the COPYING file for full license text.
 */


#include "animwin.h"

#include "model.h"
#include "decalmgr.h"
#include "log.h"
#include "msg.h"

#include "helpwin.h"

#include "mq3compat.h"

#include <qcombobox.h>
#include <qcheckbox.h>
#include <qlabel.h>
#include <qlineedit.h>
#include <qpushbutton.h>
#include <qslider.h>
#include <qspinbox.h>
#include <qtabwidget.h>
#include <qtimer.h>
#include <qinputdialog.h>

enum
{
   ANIMWIN_HELP_ID,
   ANIMWIN_UNDO_ID,
   ANIMWIN_REDO_ID
};

static int getSliderTickInterval( int val )
{
   if ( val > 25000 )
   {
      return 5000;
   }
   if ( val > 10000 )
   {
      return 1000;
   }
   if ( val > 2500 )
   {
      return 500;
   }
   if ( val > 1000 )
   {
      return 100;
   }
   if ( val > 250 )
   {
      return 50;
   }
   if ( val > 100 )
   {
      return 10;
   }
   if ( val > 25 )
   {
      return 5;
   }
   return 1;
}

AnimWindow::AnimWindow( Model * model, bool isUndo, QWidget * parent, const char * name )
   : AnimWinBase( parent, name, false, Qt::WDestructiveClose | Qt::WStyle_Tool),
     m_model( model ),
     m_doLoop( true ),
     m_playing( false ),
     m_undoing( isUndo ),
     m_ignoreChange( false ),
     m_needShutdown( true )
{
   log_debug( "AnimWindow constructor\n" );

   m_skelNew->setDefault( false );

   m_skelCountSlider->setTickmarks( QSlider::Below );
   m_frameCountSlider->setTickmarks( QSlider::Below );

   m_animTimer = new QTimer( this );
   m_accel     = new QAccel( this );

   m_accel->insertItem( Qt::Key_F1, ANIMWIN_HELP_ID );
   m_accel->insertItem( Qt::CTRL + Qt::Key_Z, ANIMWIN_UNDO_ID );
   m_accel->insertItem( Qt::CTRL + Qt::Key_Y, ANIMWIN_REDO_ID );

   connect( m_animTimer, SIGNAL(timeout()),      this, SLOT(timeElapsed()) );
   connect( m_accel,     SIGNAL(activated(int)), this, SLOT(accelActivated(int)) );

   m_doLoop = m_model->isAnimationLooping();
   m_loop->setChecked( m_doLoop );

   unsigned t;
   for ( t = 0; t < m_model->getAnimCount( Model::ANIMMODE_FRAME); t++ )
   {
      m_frameName->insertItem( m_model->getAnimName( Model::ANIMMODE_FRAME, t ), t );
   }

   for ( t = 0; t < m_model->getAnimCount( Model::ANIMMODE_SKELETAL ); t++ )
   {
      m_skelName->insertItem( m_model->getAnimName( Model::ANIMMODE_SKELETAL, t ), t );
   }

   m_skelPasteFrame->setEnabled( false );
   m_framePasteFrame->setEnabled( false );

   if ( isUndo )
   {
      bool     isSkel   = m_model->inSkeletalMode();
      unsigned anim     = m_model->getCurrentAnimation();
      unsigned frame    = m_model->getCurrentAnimationFrame();

      m_tab->setCurrentPage( m_model->inSkeletalMode() ? 0 : 1 );

      m_model->setCurrentAnimation( isSkel ? Model::ANIMMODE_SKELETAL : Model::ANIMMODE_FRAME, anim );
      m_model->setCurrentAnimationFrame( frame );

      if ( isSkel )
      {
         if ( m_skelName->count() > 0 )
         {
            m_skelName->setCurrentItem( anim );
         }
      }
      else
      {
         if ( m_frameName->count() > 0 )
         {
            m_frameName->setCurrentItem( anim );
         }
      }

      m_undoing = false;

      m_currentAnim  = anim;
      m_currentFrame = frame;
      m_isSkel       = isSkel;

      refreshPage();
   }
   else
   {
      if ( m_skelName->count() > 0 )
      {
         m_tab->setCurrentPage( 0 );
         m_model->setCurrentAnimation( Model::ANIMMODE_SKELETAL, m_skelName->currentItem() );
      }
      else if ( m_frameName->count() > 0 )
      {
         m_tab->setCurrentPage( 1 );
         m_model->setCurrentAnimation( Model::ANIMMODE_FRAME, m_frameName->currentItem() );
      }
      else
      {
         m_tab->setCurrentPage( 0 );
      }

      m_model->operationComplete( "Start animation mode" );
   }

   m_stop->setEnabled( false );
}

AnimWindow::~AnimWindow()
{
}

void AnimWindow::helpNowEvent( int )
{
   HelpWin * win = new HelpWin( "olh_animwin.html", true );
   win->show();
}

void AnimWindow::frameClearFrame()
{
   if ( m_frameName->count() > 0 && m_frameCountSlider->minValue() > 0 )
   {
      int anim  = m_frameName->currentItem();
      int frame = m_frameCountSlider->value() - 1;
      if( anim >= 0 && m_frameCountSlider->value() > 0 )
      {
         m_model->clearAnimFrame( Model::ANIMMODE_FRAME, anim, frame );
         m_model->operationComplete( "Clear frame" );
         DecalManager::getInstance()->modelUpdated( m_model );
      }
   }
}

void AnimWindow::frameCopyFrame()
{
   m_frameCopyList.clear();
   m_keyframeCopyList.clear();

   unsigned numVertices = m_model->getVertexCount();
   unsigned v = 0;

   FrameCopy copy;
   copy.vertex = 0;
   copy.x = 0;
   copy.y = 0;
   copy.z = 0;

   for ( v = 0; v < numVertices; v++ )
   {
      if ( m_model->getFrameAnimVertexCoords( m_currentAnim, m_currentFrame, v, copy.x, copy.y, copy.z ) )
      {
         copy.vertex = v;
         m_frameCopyList.push_back( copy );
      }
   }

   m_framePasteFrame->setEnabled( m_frameCopyList.size() ? true : false );
}

void AnimWindow::framePasteFrame()
{
   FrameCopyList::iterator it;

   for ( it = m_frameCopyList.begin(); it != m_frameCopyList.end(); it++ )
   {
      m_model->setFrameAnimVertexCoords( m_currentAnim, m_currentFrame, (*it).vertex,
            (*it).x, (*it).y, (*it).z );
   }
   m_model->operationComplete( "Paste frame" );

   m_model->setCurrentAnimationFrame( m_currentFrame );
   DecalManager::getInstance()->modelUpdated( m_model );
}

void AnimWindow::frameNewClicked()
{
   bool ok = false;

   QString name = QInputDialog::getText(
         tr( "Misfit 3D" ),
         tr( "New name:" ),
         QLineEdit::Normal, "", &ok, this );

   if ( ok && !name.isEmpty() )
   {
      int num = m_model->addAnimation( Model::ANIMMODE_FRAME, name.latin1() );
      if ( num >= 0 )
      {
         m_frameName->insertItem( m_model->getAnimName( Model::ANIMMODE_FRAME, num ), num );
         m_frameName->setCurrentItem( num );
         frameNameSelected( num );

         m_model->operationComplete( "New Frame Animation" );
      }
   }
}

void AnimWindow::frameRenameClicked()
{
   if ( m_frameName->count() > 0 )
   {
      bool ok = false;

      QString name = QInputDialog::getText(
            tr( "Misfit 3D" ),
            tr( "New name:" ),
            QLineEdit::Normal, m_frameName->currentText().latin1(), &ok, this );

      if ( ok && !name.isEmpty() )
      {
         m_model->setAnimName( Model::ANIMMODE_FRAME, m_frameName->currentItem(), name.latin1() );
         m_model->operationComplete( "Rename Animation" );
         m_frameName->changeItem( name, m_frameName->currentItem() );
      }
   }
}

void AnimWindow::frameDeleteClicked()
{
   if ( m_frameName->count() > 0 )
   {
      int index = m_frameName->currentItem();
      m_model->deleteAnimation( Model::ANIMMODE_FRAME, m_frameName->currentItem() );
      m_frameName->removeItem( m_frameName->currentItem() );
      m_model->operationComplete( "Delete Animation" );
      if ( m_frameName->count() > 0 )
      {
         if ( index >= m_frameName->count() )
         {
            index = 0;
         }
         m_frameName->setCurrentItem( index );
         frameNameSelected( index );
      }
      refreshPage();
   }
}

void AnimWindow::frameNameSelected( int index )
{
   log_debug( "frame name selected: %d\n", index );
   m_currentTime = 0;

   m_currentAnim  = index;
   m_currentFrame = 0;

   if( m_frameName->count() > 0 )
   {
      m_model->setCurrentAnimation( Model::ANIMMODE_FRAME, m_currentAnim );
   }
   else
   {
      m_model->setCurrentAnimation( Model::ANIMMODE_FRAME, (unsigned) 0 );
   }

   refreshPage();
}

void AnimWindow::frameSetCurrentFrame( int frame )
{
   if ( m_model->setCurrentAnimationFrame( frame - 1 ) )
   {
      m_currentFrame = frame - 1;
      QString str;
      str.sprintf( "Frame: %03d", frame );
      m_frameCountLabel->setText( str );
   }
   else
   {
      m_frameCountLabel->setText( "Frame: n/a" );
   }
   DecalManager::getInstance()->modelUpdated( m_model );
}

void AnimWindow::frameChangeFPS()
{
   if ( m_frameName->count() > 0 )
   {
      log_debug( "changing FPS\n" );
      m_model->setAnimFPS( Model::ANIMMODE_FRAME, m_frameName->currentItem(), atof(m_frameFPS->text().latin1()) );
      m_model->operationComplete( "Set FPS" );
   }
}

void AnimWindow::frameChangeFrameCount()
{
   if ( !m_ignoreChange )
   {
      if ( m_frameName->count() > 0 )
      {
         m_model->setAnimFrameCount( Model::ANIMMODE_FRAME, m_frameName->currentItem(), m_frameCount->value() );
         m_model->operationComplete( "Change Frame Count" );
         m_frameCountSlider->setMinValue( 1 );
         m_frameCountSlider->setValue( m_model->getCurrentAnimationFrame() + 1 );
         frameSetCurrentFrame( m_model->getCurrentAnimationFrame() + 1 );
      }
      m_frameCountSlider->setMaxValue( m_frameCount->value() );
      m_frameCountSlider->update();
      DecalManager::getInstance()->modelAnimate( m_model );
   }
}

void AnimWindow::skelClearFrame()
{
   if ( m_skelName->count() > 0 && m_skelCountSlider->minValue() > 0 )
   {
      int anim  = m_skelName->currentItem();
      int frame = m_skelCountSlider->value() - 1;
      if( anim >= 0 && m_skelCountSlider->value() >= 0 )
      {
         m_model->clearAnimFrame( Model::ANIMMODE_SKELETAL, anim, frame );
         m_model->operationComplete( "Clear keyframe" );
         DecalManager::getInstance()->modelUpdated( m_model );
      }
   }
}

void AnimWindow::skelCopyFrame()
{
   m_frameCopyList.clear();
   m_keyframeCopyList.clear();

   unsigned numJoints = m_model->getBoneJointCount();
   unsigned j = 0;

   KeyframeCopy copy;
   copy.joint = 0;
   copy.x = 0;
   copy.y = 0;
   copy.z = 0;
   copy.isRotation = false;

   for ( j = 0; j < numJoints; j++ )
   {
      if ( m_model->getSkelAnimKeyframe( m_currentAnim, m_currentFrame, j, false, copy.x, copy.y, copy.z ) )
      {
         copy.joint = j;
         copy.isRotation = false;

         m_keyframeCopyList.push_back( copy );
      }
   }

   for ( j = 0; j < numJoints; j++ )
   {
      if ( m_model->getSkelAnimKeyframe( m_currentAnim, m_currentFrame, j, true, copy.x, copy.y, copy.z ) )
      {
         copy.joint = j;
         copy.isRotation = true;

         m_keyframeCopyList.push_back( copy );
      }
   }

   m_skelPasteFrame->setEnabled( m_keyframeCopyList.size() ? true : false );
}

void AnimWindow::skelPasteFrame()
{
   KeyframeCopyList::iterator it;

   for ( it = m_keyframeCopyList.begin(); it != m_keyframeCopyList.end(); it++ )
   {
      m_model->setSkelAnimKeyframe( m_currentAnim, m_currentFrame, (*it).joint, (*it).isRotation,
            (*it).x, (*it).y, (*it).z );
   }
   m_model->operationComplete( "Paste keyframe" );

   m_model->setCurrentAnimationFrame( m_currentFrame );  // Force refresh of joints
   DecalManager::getInstance()->modelUpdated( m_model );
}

void AnimWindow::skelNewClicked()
{
   bool ok = false;

   QString name = QInputDialog::getText(
         tr( "Misfit 3D" ),
         tr( "New name:" ),
         QLineEdit::Normal, "", &ok, this );

   if ( ok && !name.isEmpty() )
   {
      int num = m_model->addAnimation( Model::ANIMMODE_SKELETAL, name.latin1() );
      if ( num >= 0 )
      {
         m_skelName->insertItem( name.latin1(), num );
         m_skelName->setCurrentItem( num );
         skelNameSelected( num );

         m_model->operationComplete( "New Skeletal Animation" );
      }
   }
}

void AnimWindow::skelRenameClicked()
{
   if ( m_skelName->count() > 0 )
   {
      bool ok = false;

      QString name = QInputDialog::getText(
            tr( "Misfit 3D" ),
            tr( "New name:" ),
            QLineEdit::Normal, m_skelName->currentText().latin1(), &ok, this );

      if ( ok && !name.isEmpty() )
      {
         m_model->setAnimName( Model::ANIMMODE_SKELETAL, m_skelName->currentItem(), name.latin1() );
         m_model->operationComplete( "Rename Animation" );
         m_skelName->changeItem( name, m_skelName->currentItem() );
      }
   }
}

void AnimWindow::skelDeleteClicked()
{
   if ( m_skelName->count() > 0 )
   {
      int index = m_skelName->currentItem();
      m_model->deleteAnimation( Model::ANIMMODE_SKELETAL, m_skelName->currentItem() );
      m_skelName->removeItem( m_skelName->currentItem() );
      m_model->operationComplete( "Delete Animation" );
      if ( m_skelName->count() > 0 )
      {
         if ( index >= m_skelName->count() )
         {
            index = 0;
         }
         m_skelName->setCurrentItem( index );
         skelNameSelected( index );
      }
      refreshPage();
   }
}

void AnimWindow::skelNameSelected( int index )
{
   log_debug( "skel name selected: %d\n", index );
   m_currentTime = 0;

   m_currentAnim  = index;
   m_currentFrame = 0;

   if( m_skelName->count() > 0 )
   {
      m_model->setCurrentAnimation( Model::ANIMMODE_SKELETAL, m_currentAnim );

      list<int> joints = m_model->getSelectedBoneJoints();
   }
   else
   {
      m_model->setCurrentAnimation( Model::ANIMMODE_SKELETAL, (unsigned) 0 );
   }

   refreshPage();
}

void AnimWindow::skelSetCurrentFrame( int frame )
{
   LOG_PROFILE();
   {
      LOG_PROFILE_STR( "set current frame" );
      if ( m_model->setCurrentAnimationFrame( frame - 1 ) )
      {
         m_currentFrame = frame - 1;
         QString str;
         str.sprintf( "Frame: %03d", frame );
         m_skelCountLabel->setText( str );
      }
      else
      {
         m_skelCountLabel->setText( "Frame: n/a" );
      }
   }
   {
      LOG_PROFILE_STR( "model updated" );
      DecalManager::getInstance()->modelUpdated( m_model );
   }
}

void AnimWindow::skelChangeFPS()
{
   if ( m_skelName->count() > 0 )
   {
      m_model->setAnimFPS( Model::ANIMMODE_SKELETAL, m_skelName->currentItem(), atof(m_skelFPS->text().latin1()) );
      m_model->operationComplete( "Set FPS" );
   }
}

void AnimWindow::skelChangeFrameCount()
{
   if ( !m_ignoreChange )
   {
      if ( m_skelName->count() > 0 )
      {
         m_model->setAnimFrameCount( Model::ANIMMODE_SKELETAL, m_skelName->currentItem(), m_skelCount->value() );
         m_model->operationComplete( "Change Frame Count" );
         m_skelCountSlider->setMinValue( 1 );
         m_skelCountSlider->setValue( m_model->getCurrentAnimationFrame() + 1 );
         skelSetCurrentFrame( m_model->getCurrentAnimationFrame() + 1 );
      }
      m_skelCountSlider->setMaxValue( m_skelCount->value() );
      m_skelCountSlider->update();
      DecalManager::getInstance()->modelAnimate( m_model );
   }
}

void AnimWindow::playClicked()
{
   if ( m_playing )
   {
      doPause();
   }
   else
   {
      doPlay();
   }
   m_stop->setEnabled( true );
}

void AnimWindow::stopClicked()
{
   m_currentTime = 0;
   m_stop->setEnabled( false );
   m_play->setText("Play");
   m_playing = false;
   m_animTimer->stop();
   m_model->setCurrentAnimationFrame( ((m_isSkel) ? m_skelCountSlider->value() : m_frameCountSlider->value()) - 1 );
   DecalManager::getInstance()->modelUpdated( m_model );
}

void AnimWindow::loopToggled( bool o )
{
   m_doLoop = o;
   m_model->setAnimationLooping( o );
}

void AnimWindow::doPlay()
{
   m_playing = true;
   m_play->setText("Pause");
   if ( m_isSkel )
   {
      if ( m_skelName->count() == 0 )
      {
         return;
      }
      m_timeInterval = double (1.0 / m_model->getAnimFPS( Model::ANIMMODE_SKELETAL, m_skelName->currentItem() ));
   }
   else
   {
      if ( m_frameName->count() == 0 )
      {
         return;
      }
      m_timeInterval = double (1.0 / m_model->getAnimFPS( Model::ANIMMODE_FRAME, m_frameName->currentItem() ));
   }

   PORT_gettimeofday( &m_startTime );

   m_animTimer->start( (int) (m_timeInterval * 1000), false );
   log_debug( "starting %s animation, update every %.03f seconds\n", (m_isSkel ? "skeletal" : "frame" ), m_timeInterval );
}

void AnimWindow::doPause()
{
   m_playing = false;
   m_play->setText("Play");
   m_animTimer->stop();
}

void AnimWindow::reject()
{
   log_debug( "closing window on reject\n" );
   emit animWindowClosed();
   stopAnimationMode();
   return AnimWinBase::reject();
}

void AnimWindow::closeEvent( QCloseEvent * e )
{
   AnimWinBase::closeEvent( e );
   emit animWindowClosed();
   stopAnimationMode();
}

void AnimWindow::timeElapsed()
{
   PORT_timeval tv;
   PORT_gettimeofday( &tv );
   unsigned t = (tv.tv_sec - m_startTime.tv_sec) * 1000 + (tv.tv_msec - m_startTime.tv_msec);

   m_currentTime = ((double) t) / 1000.0;
   if ( m_model->setCurrentAnimationTime( m_currentTime ) )
   {
      //log_debug( "animation time: %f (%d)\n", m_currentTime, t );
   }
   else
   {
      stopClicked();
      //log_debug( "animation time: %f (complete)\n", m_currentTime );
   }
   DecalManager::getInstance()->modelAnimate( m_model );
}

void AnimWindow::pageChanged( QWidget * newPage )
{
   log_debug( "anim window page changed\n" );

   if ( !m_undoing )
   {
      int index = m_tab->indexOf( newPage );

      bool wasSkel = m_isSkel;

      m_isSkel = (index == 0) ? true : false;

      stopClicked();

      if ( wasSkel != m_isSkel )
      {
         if ( m_isSkel )
         {
            log_debug( "Using skeletal animation\n" );
            m_skelCountSlider->setMinValue( 1 );
            if ( m_skelName->count() > 0 )
            {
               m_skelName->setCurrentItem( 0 );
            }
            skelNameSelected( 0 );
         }
         else
         {
            log_debug( "Using frame animation\n" );
            m_frameCountSlider->setMinValue( 1 );
            if ( m_frameName->count() )
            {
               m_frameName->setCurrentItem( 0 );
            }
            frameNameSelected( 0 );
         }
      }
   }
}

void AnimWindow::undoRequest()
{
   log_debug( "anim undo request\n" );

   m_model->undo();

   if ( !m_model->getAnimationMode() )
   {
      close();
      return;
   }

   m_undoing = true;

   bool     isSkel = m_model->inSkeletalMode();
   int      page   = (isSkel) ? 0 : 1;
   unsigned anim   = m_model->getCurrentAnimation();
   unsigned frame  = m_model->getCurrentAnimationFrame();

   if ( m_tab->currentPageIndex() != page )
   {
      m_tab->setCurrentPage( page );
      m_isSkel = isSkel;
   }

   QComboBox * name   = NULL;

   if ( page == 0 )
   {
      name = m_skelName;
      int numFrames = m_model->getAnimCount( Model::ANIMMODE_SKELETAL );
      int t;

      for ( t = 0; t < name->count() && t < numFrames; t++ )
      {
         name->changeItem( m_model->getAnimName( Model::ANIMMODE_SKELETAL, t), t );
      }
      
      for ( t = name->count(); t < numFrames; t++ )
      {
         name->insertItem( m_model->getAnimName(  Model::ANIMMODE_SKELETAL, t ), t );
      }

      while( (int) name->count() > numFrames )
      {
         name->removeItem( numFrames );
      }
   }
   else
   {
      name = m_frameName;
      int numFrames = m_model->getAnimCount( Model::ANIMMODE_FRAME );
      int t;

      for ( t = 0; t < name->count() && t < numFrames; t++ )
      {
         name->changeItem( m_model->getAnimName( Model::ANIMMODE_FRAME, t), t );
      }
      
      for ( t = name->count(); t < numFrames; t++ )
      {
         name->insertItem( m_model->getAnimName(  Model::ANIMMODE_FRAME, t ), t );
      }

      while( (int) name->count() > numFrames )
      {
         name->removeItem( numFrames );
      }
   }

   name->setCurrentItem( anim );
   m_model->setCurrentAnimation( isSkel ? Model::ANIMMODE_SKELETAL : Model::ANIMMODE_FRAME, anim );
   m_model->setCurrentAnimationFrame( frame );
   m_currentAnim  = anim;
   m_currentFrame = frame;

   m_undoing = false;

   refreshPage();
   DecalManager::getInstance()->modelUpdated( m_model );
}

void AnimWindow::redoRequest()
{
   log_debug( "anim redo request\n" );

   m_model->redo();

   if ( !m_model->getAnimationMode() )
   {
      close();
      return;
   }

   m_undoing = true;

   bool     isSkel = m_model->inSkeletalMode();
   int      page   = (isSkel) ? 0 : 1;
   unsigned anim   = m_model->getCurrentAnimation();
   unsigned frame  = m_model->getCurrentAnimationFrame();

   if ( m_tab->currentPageIndex() != page )
   {
      m_tab->setCurrentPage( page );
      m_isSkel = isSkel;
   }

   QComboBox * name   = NULL;

   if ( page == 0 )
   {
      name = m_skelName;
      int numFrames = m_model->getAnimCount( Model::ANIMMODE_SKELETAL );
      int t;

      for ( t = 0; t < name->count() && t < numFrames; t++ )
      {
         name->changeItem( m_model->getAnimName( Model::ANIMMODE_SKELETAL, t), t );
      }
      
      for ( t = name->count(); t < numFrames; t++ )
      {
         name->insertItem( m_model->getAnimName(  Model::ANIMMODE_SKELETAL, t ), t );
      }

      while( (int) name->count() > numFrames )
      {
         name->removeItem( numFrames );
      }
   }
   else
   {
      name = m_frameName;
      int numFrames = m_model->getAnimCount( Model::ANIMMODE_FRAME );
      int t;

      for ( t = 0; t < name->count() && t < numFrames; t++ )
      {
         name->changeItem( m_model->getAnimName( Model::ANIMMODE_FRAME, t), t );
      }
      
      for ( t = name->count(); t < numFrames; t++ )
      {
         name->insertItem( m_model->getAnimName(  Model::ANIMMODE_FRAME, t ), t );
      }

      while( (int) name->count() > numFrames )
      {
         name->removeItem( numFrames );
      }
   }

   name->setCurrentItem( anim );
   m_model->setCurrentAnimation( isSkel ? Model::ANIMMODE_SKELETAL : Model::ANIMMODE_FRAME, anim );
   m_model->setCurrentAnimationFrame( frame );
   m_currentAnim  = anim;
   m_currentFrame = frame;

   m_undoing = false;

   refreshPage();
   DecalManager::getInstance()->modelUpdated( m_model );
}

void AnimWindow::accelActivated( int id )
{
   switch ( id )
   {
      case ANIMWIN_HELP_ID:
         helpNowEvent( id );
         break;
      case ANIMWIN_UNDO_ID:
         undoRequest();
         break;
      case ANIMWIN_REDO_ID:
         redoRequest();
         break;
      default:
         log_error( "Unknown animwindow accel id: %d", id );
         break;
   }
}

void AnimWindow::refreshPage()
{
   log_debug( "refresh anim window page\n" );

   if ( !m_undoing )
   {
      if ( m_isSkel )
      {
         if( m_skelName->count() > 0 )
         {
            int index = m_skelName->currentItem();

            m_skelCount->setEnabled( true );
            m_skelFPS->setEnabled( true );
            m_skelName->setEnabled( true );
            m_skelCountSlider->setEnabled( true );
            m_skelCopyFrame->setEnabled( true );
            m_skelPasteFrame->setEnabled( m_keyframeCopyList.size() ? true : false );
            m_skelClearFrame->setEnabled( true );
            m_skelDelete->setEnabled( true );
            m_skelRename->setEnabled( true );
            m_play->setEnabled( true );

            unsigned count = m_model->getAnimFrameCount( Model::ANIMMODE_SKELETAL, index );

            m_ignoreChange = true;  // Qt alerts us even if we're responsible
            m_skelCount->setValue( count );
            m_ignoreChange = false;

            m_skelFPS->setText( QString::number(m_model->getAnimFPS( Model::ANIMMODE_SKELETAL, index ) ) );
            m_skelCountSlider->setMinValue( 1 );
            m_skelCountSlider->setMaxValue( count );
            m_skelCountSlider->setValue( m_currentFrame + 1 );
            m_skelCountSlider->setTickInterval( getSliderTickInterval( count ) );
            m_skelCountSlider->update();

            skelSetCurrentFrame( m_currentFrame + 1 );
         }
         else
         {
            m_skelCount->setEnabled( false );
            m_skelFPS->setEnabled( false );
            m_skelName->setEnabled( false );
            m_skelCountSlider->setEnabled( false );
            m_skelCopyFrame->setEnabled( false );
            m_skelPasteFrame->setEnabled( false );
            m_skelClearFrame->setEnabled( false );
            m_skelDelete->setEnabled( false );
            m_skelRename->setEnabled( false );
            m_play->setEnabled( false );
            m_stop->setEnabled( false );

            m_skelFPS->setText( "0" );
            m_ignoreChange = true;  // Qt alerts us even if we're responsible
            m_skelCount->setValue( 0 );
            m_ignoreChange = false;
            m_skelCountSlider->setMinValue( 0 );
            m_skelCountSlider->setMaxValue( 0 );
            m_skelCountSlider->setValue( 0 );
            skelSetCurrentFrame( 0 );
         }
      }
      else
      {
         if ( m_frameName->count() > 0 )
         {
            int index = m_frameName->currentItem();

            m_frameCount->setEnabled( true );
            m_frameFPS->setEnabled( true );
            m_frameName->setEnabled( true );
            m_frameCountSlider->setEnabled( true );
            m_frameCopyFrame->setEnabled( true );
            m_framePasteFrame->setEnabled( m_frameCopyList.size() ? true : false );
            m_frameClearFrame->setEnabled( true );
            m_frameDelete->setEnabled( true );
            m_frameRename->setEnabled( true );
            m_play->setEnabled( true );

            unsigned count = m_model->getAnimFrameCount( Model::ANIMMODE_FRAME, index );

            m_ignoreChange = true;  // Qt alerts us even if we're responsible
            m_frameCount->setValue( count );
            m_ignoreChange = false;

            m_frameFPS->setText( QString::number(m_model->getAnimFPS( Model::ANIMMODE_FRAME, index ) ) );
            m_frameCountSlider->setMinValue( 1 );
            m_frameCountSlider->setMaxValue( count );
            m_frameCountSlider->setValue( m_currentFrame + 1 );
            m_frameCountSlider->setTickInterval( getSliderTickInterval( count ) );
            m_frameCountSlider->update();

            frameSetCurrentFrame( m_currentFrame + 1 );
         }
         else
         {
            m_frameCount->setEnabled( false );
            m_frameFPS->setEnabled( false );
            m_frameName->setEnabled( false );
            m_frameCountSlider->setEnabled( false );
            m_frameCopyFrame->setEnabled( false );
            m_framePasteFrame->setEnabled( false );
            m_frameClearFrame->setEnabled( false );
            m_frameDelete->setEnabled( false );
            m_frameRename->setEnabled( false );
            m_play->setEnabled( false );
            m_stop->setEnabled( false );

            m_frameFPS->setText( "0" );
            m_ignoreChange = true;  // Qt alerts us even if we're responsible
            m_frameCount->setValue( 0 );
            m_ignoreChange = false;
            m_frameCountSlider->setMinValue( 0 );
            m_frameCountSlider->setMaxValue( 0 );
            m_frameCountSlider->setValue( 0 );
            frameSetCurrentFrame( 0 );
         }
      }
   }
   else
   {
      if ( m_isSkel )
      {
         if ( m_skelName->count() == 0)
         {
            m_skelCount->setEnabled( false );
            m_skelFPS->setEnabled( false );
            m_skelName->setEnabled( false );
            m_skelCountSlider->setEnabled( false );
            m_skelCopyFrame->setEnabled( false );
            m_skelPasteFrame->setEnabled( false );
            m_skelClearFrame->setEnabled( false );
            m_skelDelete->setEnabled( false );
            m_skelRename->setEnabled( false );
            m_play->setEnabled( false );
            m_stop->setEnabled( false );

            m_skelFPS->setText( "0" );
            m_ignoreChange = true;  // Qt alerts us even if we're responsible
            m_skelCount->setValue( 0 );
            m_ignoreChange = false;
            m_skelCountSlider->setMinValue( 0 );
            m_skelCountSlider->setMaxValue( 0 );
            m_skelCountSlider->setValue( 0 );
            skelSetCurrentFrame( 0 );
         }
      }
      else
      {
         if ( m_frameName->count() == 0 )
         {
            m_frameCount->setEnabled( false );
            m_frameFPS->setEnabled( false );
            m_frameName->setEnabled( false );
            m_frameCountSlider->setEnabled( false );
            m_frameCopyFrame->setEnabled( false );
            m_framePasteFrame->setEnabled( false );
            m_frameClearFrame->setEnabled( false );
            m_frameDelete->setEnabled( false );
            m_frameRename->setEnabled( false );
            m_play->setEnabled( false );
            m_stop->setEnabled( false );

            m_frameFPS->setText( "0" );
            m_ignoreChange = true;  // Qt alerts us even if we're responsible
            m_frameCount->setValue( 0 );
            m_ignoreChange = false;
            m_frameCountSlider->setMinValue( 0 );
            m_frameCountSlider->setMaxValue( 0 );
            m_frameCountSlider->setValue( 0 );
            frameSetCurrentFrame( 0 );
         }
      }
   }
}

void AnimWindow::stopAnimationMode()
{
   if ( m_needShutdown )
   {
      if ( m_model->getAnimationMode() ) 
      {
         m_model->setNoAnimation();
         m_model->operationComplete( "End animation mode" );
         DecalManager::getInstance()->modelUpdated( m_model );
      }
   }
   m_needShutdown = false;
}

