/*  Misfit Model 3D
 * 
 *  Copyright (c) 2004-2005 Kevin Worcester
 * 
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, 
 *  USA.
 *
 *  See the COPYING file for full license text.
 */


#include "config.h"
#include "pluginmgr.h"
#include "viewwin.h"
#include "viewpanel.h"
#include "model.h"
#include "toolbox.h"
#include "cmdmgr.h"
#include "groupwin.h"
#include "texwin.h"
#include "texturecoord.h"
#include "log.h"
#include "decalmgr.h"
#include "msg.h"
#include "statusbar.h"
#include "modelstatus.h"
#include "filtermgr.h"
#include "misc.h"
#include "helpwin.h"
#include "licensewin.h"
#include "aboutwin.h"
#include "animsetwin.h"
#include "animexportwin.h"
#include "animwin.h"
#include "metawin.h"
#include "3dmprefs.h"
#include "stdtools.h"
#include "stdcmds.h"
#include "pluginwin.h"
#include "backgroundwin.h"
#include "mergewin.h"
#include "misc.h"
#include "checkupdate.h"
#include "version.h"
#include "texmgr.h"

#include "qtmain.h"

#include "luascript.h"
#include "luaif.h"

#include "pixmap/mm3dlogo-32x32.xpm"

#include "mq3macro.h"
#include "mq3compat.h"
#include "memguard.h"

#include <qapplication.h>
#include <qlayout.h>
#include <qmenubar.h>
#include <qmessagebox.h>
#include <stdio.h>
#include <qtoolbutton.h>
#include <qtooltip.h>
#include <qtimer.h>
#ifdef HAVE_QT4
#include <q3toolbar.h>
#else
#include <qtoolbar.h>
#endif

#ifdef HAVE_QT4
#include <QCloseEvent>
#endif // HAVE_QT4

#include <list>
#include <string>

using std::list;
using std::string;

#ifdef HAVE_QT4
#define setSelectedFilter selectFilter
#endif

//using namespace QIconSet;

typedef QToolButton * QToolButtonPtr;
typedef ::Tool * ToolPtr;

typedef std::list< ViewWindow *> ViewWindowList;
static ViewWindowList _winList;

static bool _shuttingDown = false;

static void _versionNotify( Model * model )
{
   int best = -1;
   int bmajor = VERSION_MAJOR;
   int bminor = VERSION_MINOR;
   int bpatch = VERSION_PATCH;
   const char * bstr = VERSION_STRING;

   int major = 0;
   int minor = 0;
   int patch = 0;
   const char * str = VERSION_STRING;

   str = check_update_get_version_string( CUV_Stable );
   check_update_get_version( CUV_Stable, major, minor, patch );

   if ( major > bmajor || 
         ((major == bmajor) && 
          ((minor > bminor) || 
           ((minor == bminor) && patch > bpatch ) ) ) )
   {
      best = CUV_Stable;
      bmajor = major;
      bminor = minor;
      bpatch = patch;
      bstr = str;
   }

   str = check_update_get_version_string( CUV_Devel );
   check_update_get_version( CUV_Devel, major, minor, patch );

   if ( major > bmajor || 
         ((major == bmajor) && 
          ((minor > bminor) || 
           ((minor == bminor) && patch > bpatch ) ) ) )
   {
      best = CUV_Devel;
      bmajor = major;
      bminor = minor;
      bpatch = patch;
      bstr = str;
   }

   if ( best != -1 )
   {
      char s[80];
      PORT_snprintf( s, sizeof(s), "%s version %s available", 
            (best == CUV_Stable) ? "Stable" : "Development", bstr );
      model_status( model, StatusNormal, STATUSTIME_SHORT, s );
   }
}

bool ViewWindow::closeAllWindows()
{
   bool noPrompt = true;
   bool doSave   = false;

   std::list<ViewWindow *>::iterator it;

   for ( it = _winList.begin(); noPrompt == true && it != _winList.end(); it++ )
   {
      noPrompt = (*it)->getSaved();
   }

#ifndef CODE_DEBUG
   if ( noPrompt == false )
   {
       char response = msg_warning_prompt( "Some models are unsaved.  Save before exiting?" );

       if ( response == 'C' )
       {
          return false;
       }

       if ( response == 'Y' )
       {
          doSave = true;
       }
   }
#endif // CODE_DEBUG

   bool abortQuit = false;

   for ( it = _winList.begin(); !abortQuit && it != _winList.end(); it++ )
   {
      ViewWindow * win = (*it);

      if ( doSave && win->getSaved() == false )
      {
         win->raise();
         win->setAbortQuit( false );
         win->saveModelEvent();
         abortQuit = win->getAbortQuit();
      }
   }

   if ( !abortQuit )
   {
      while ( !_winList.empty() )
      {
         ViewWindow * win = _winList.front();

         win->hide();
         win->deleteLater();

         _winList.pop_front();
      }
   }

   if ( abortQuit )
   {
      return false;
   }
   else
   {
      return true;
   }
}

class MainWidget : public QWidget
{
   public:
      MainWidget( QWidget * parent = NULL, const char * name = "" );
      virtual ~MainWidget() {};

      void addWidgetToLayout( QWidget * w );

   protected:

      QVBoxLayout * m_layout;
};

MainWidget::MainWidget( QWidget * parent, const char * name )
   : QWidget( parent, name )
{
   m_layout = new QVBoxLayout( this );
}

void MainWidget::addWidgetToLayout( QWidget * w )
{
   m_layout->addWidget( w );
}

ViewWindow::ViewWindow( Model * model, QWidget * parent, const char * name )
   : QMainWindow( parent, name, WDestructiveClose ),
     m_accel( new QAccel(this) ),
     m_model( model ),
     m_animWin( NULL ),
     m_toolbox( NULL ),
     m_toolList( NULL ),
     m_toolButtons( NULL ),
     m_last( NULL ),
     m_currentTool( NULL )
{
   _winList.push_back( this );

   setIcon( (const char **) mm3dlogo_32x32_xpm );
   setIconText( "Misfit Model 3D" );

   m_accel->insertItem( Key_F1, 0 );
   connect( m_accel, SIGNAL(activated(int)), this, SLOT(helpNowEvent(int)) );

   updateCaption();

   m_toolbox = new Toolbox();

   MainWidget * mainWidget = new MainWidget( this );

   m_viewPanel = new ViewPanel( m_toolbox, mainWidget, "" );
   mainWidget->addWidgetToLayout( m_viewPanel );

   m_statusBar = new StatusBar( m_model, mainWidget, "" );
   mainWidget->addWidgetToLayout( m_statusBar );
   m_statusBar->setText("Press F1 for help using any window");
   m_statusBar->setMaximumHeight( 30 );

   setModel( m_model );

   setCentralWidget( mainWidget );

   m_mruMenu = new QPopupMenu( this, "" );
   m_scriptMruMenu = new QPopupMenu( this, "" );
   connect( m_mruMenu, SIGNAL(aboutToShow()), this, SLOT(fillMruMenu()) );
   connect( m_mruMenu, SIGNAL(activated(int)), this, SLOT(openMru(int)) );
   connect( m_scriptMruMenu, SIGNAL(aboutToShow()), this, SLOT(fillScriptMruMenu()) );
   connect( m_scriptMruMenu, SIGNAL(activated(int)), this, SLOT(openScriptMru(int)) );

   m_fileMenu = new QPopupMenu( this );
   m_fileMenu->insertItem( "New", this, SLOT(newModelEvent()), CTRL + Key_N );
   m_fileMenu->insertItem( "Open", this, SLOT(openModelEvent()), CTRL + Key_O );
   m_fileMenu->insertItem( "Save", this, SLOT(saveModelEvent()), CTRL+Key_S );
   m_fileMenu->insertItem( "Save As", this, SLOT(saveModelAsEvent()) );
   m_fileMenu->insertSeparator();
   m_fileMenu->insertItem( "Set Background Image...", this, SLOT(backgroundWindowEvent()) );
#ifdef HAVE_LUALIB
   m_fileMenu->insertItem( "Run Script...", this, SLOT(scriptEvent()) );
   m_fileMenu->insertItem( "Recent scripts", m_scriptMruMenu );
#endif // HAVE_LUALIB
   m_fileMenu->insertItem( "Merge...", this, SLOT(mergeModelsEvent()) );
   m_fileMenu->insertItem( "Merge Animations...", this, SLOT(mergeAnimationsEvent()) );
   m_fileMenu->insertSeparator();
   m_fileMenu->insertItem( "Recent models", m_mruMenu );
   m_fileMenu->insertSeparator();
   m_fileMenu->insertItem( "Plugins...", this, SLOT(pluginWindowEvent()) );
   m_fileMenu->insertSeparator();
   m_fileMenu->insertItem( "Close", this, SLOT(close()));
   m_fileMenu->insertItem( "Quit", this, SLOT(quitEvent()), CTRL + Key_Q );
   
   m_viewMenu = new QPopupMenu( this );
   m_viewMenu->insertItem( "Frame All", this, SLOT(frameAllEvent()), Key_Home);
   m_viewMenu->insertItem( "Frame Selected", this, SLOT(frameSelectedEvent()), SHIFT+Key_Home );
   m_viewMenu->insertSeparator();
   m_viewMenu->insertItem( "3D Wireframe",   m_viewPanel, SLOT(wireframeEvent()), CTRL+SHIFT+Key_W );
   m_viewMenu->insertItem( "3D Flat",        m_viewPanel, SLOT(flatEvent()),      CTRL+SHIFT+Key_F );
   m_viewMenu->insertItem( "3D Smooth",      m_viewPanel, SLOT(smoothEvent()),    CTRL+SHIFT+Key_S );
   m_viewMenu->insertItem( "3D Texture",     m_viewPanel, SLOT(textureEvent()),   CTRL+SHIFT+Key_T );
   m_viewMenu->insertItem( "3D Alpha Blend", m_viewPanel, SLOT(alphaEvent()),     CTRL+SHIFT+Key_A );
   m_viewMenu->insertSeparator();
   m_viewMenu->insertItem( "Canvas Wireframe",   m_viewPanel, SLOT(canvasWireframeEvent()) );
   m_viewMenu->insertItem( "Canvas Flat",        m_viewPanel, SLOT(canvasFlatEvent()) );
   m_viewMenu->insertItem( "Canvas Smooth",      m_viewPanel, SLOT(canvasSmoothEvent()) );
   m_viewMenu->insertItem( "Canvas Texture",     m_viewPanel, SLOT(canvasTextureEvent()) );
   m_viewMenu->insertItem( "Canvas Alpha Blend", m_viewPanel, SLOT(canvasAlphaEvent()) );
   m_viewMenu->insertSeparator();
   m_viewMenu->insertItem( "Hide Joints",      m_viewPanel, SLOT(boneJointHide()) );
   m_viewMenu->insertItem( "Draw Joint Lines", m_viewPanel, SLOT(boneJointLines()) );
   m_viewMenu->insertItem( "Draw Joint Bones", m_viewPanel, SLOT(boneJointBones()) );
   m_viewMenu->insertSeparator();
   if ( !g_prefs.exists( "ui_render_bad_textures" ) )
   {
      g_prefs( "ui_render_bad_textures" ) = 1;
   }
   int renderBad = g_prefs( "ui_render_bad_textures" );
   m_renderBadItem = m_viewMenu->insertItem( renderBad ? "Hide Bad Textures" : "Render Bad Textures",   this, SLOT(renderBadEvent()) );
   if ( !g_prefs.exists( "ui_render_3d_selections" ) )
   {
      g_prefs( "ui_render_render_3d_selections" ) = 0;
   }
   int renderSelection = g_prefs( "ui_render_3d_selections" );
   m_renderSelectionItem = m_viewMenu->insertItem( renderSelection ? "Hide 3D Selection" : "Render 3D Selection",   this, SLOT(renderSelectionEvent()) );
   
   m_viewMenu->insertSeparator();
   m_viewMenu->insertItem( "1 View",   m_viewPanel, SLOT(view1())   );
   m_viewMenu->insertItem( "1x2 View", m_viewPanel, SLOT(view1x2()) );
   m_viewMenu->insertItem( "2x1 View", m_viewPanel, SLOT(view2x1()) );
   m_viewMenu->insertItem( "2x2 View", m_viewPanel, SLOT(view2x2()) );
   m_viewMenu->insertItem( "2x3 View", m_viewPanel, SLOT(view2x3()) );
   m_viewMenu->insertItem( "3x2 View", m_viewPanel, SLOT(view3x2()) );
   m_viewMenu->insertItem( "3x3 View", m_viewPanel, SLOT(view3x3()) );
   
   m_toolMenu = new QPopupMenu( this );
   m_toolMenu->insertItem( "Undo", this, SLOT(undoRequest()), CTRL+Key_Z);
   m_toolMenu->insertItem( "Redo", this, SLOT(redoRequest()), CTRL+Key_Y);
   m_toolMenu->insertSeparator();

#ifdef HAVE_QT4
   m_toolBar = new Q3ToolBar( this );
   addToolBar( m_toolBar );
#else
   m_toolBar = new QToolBar( this );
#endif 
   //m_toolBar->setHorizontallyStretchable( true );
   //m_toolBar->setVerticallyStretchable( true );

   initializeToolbox();

   m_primitiveMenu = new QPopupMenu( this );
   m_groupMenu     = new QPopupMenu( this );

   m_groupMenu->insertItem( "Edit Groups...", this, SLOT(groupWindowEvent()), CTRL+Key_G );
   m_groupMenu->insertItem( "Edit Materials...", this, SLOT(textureWindowEvent()), CTRL+Key_M );
   m_groupMenu->insertItem( "Edit Texture Coordinates...", this, SLOT(textureCoordEvent()), CTRL+Key_E );
   m_groupMenu->insertItem( "Edit Model Meta Data...", this, SLOT(metaWindowEvent()) );
   m_groupMenu->insertItem( "Reload Textures", this, SLOT(reloadTexturesEvent()) );

   initializeCommands();

   connect( m_primitiveMenu, SIGNAL(activated(int)), this, SLOT(primitiveCommandActivated(int)) );
   connect( m_groupMenu,     SIGNAL(activated(int)), this, SLOT(groupCommandActivated(int))     );

   //m_scriptMenu = new QPopupMenu( this );

   m_animMenu = new QPopupMenu( this );
   m_startAnimItem     = m_animMenu->insertItem( "Start Animation Mode...", this, SLOT(startAnimationMode()) );
   m_stopAnimItem      = m_animMenu->insertItem( "Stop Animation Mode",     this, SLOT(stopAnimationMode()) );
   m_animMenu->insertSeparator();
   m_animSetsItem      = m_animMenu->insertItem( "Animation Sets...", this, SLOT(animSetWindowEvent()) );
   m_animExportItem    = m_animMenu->insertItem( "Export Animation...", this, SLOT(animExportWindowEvent()) );
   m_animMenu->insertSeparator();
   m_animSetRotItem    = m_animMenu->insertItem( "Set Rotation Keyframe", this, SLOT(animSetRotEvent()) );
   m_animSetTransItem  = m_animMenu->insertItem( "Set Translation Keyframe", this, SLOT(animSetTransEvent()) );

   m_animMenu->setItemEnabled( m_stopAnimItem, false );
   m_animMenu->setItemEnabled( m_animSetRotItem, false );
   m_animMenu->setItemEnabled( m_animSetTransItem, false );

   m_helpMenu = new QPopupMenu( this );
   m_helpMenu->insertItem( "Contents...", this, SLOT(helpWindowEvent()) );
   m_helpMenu->insertItem( "License...",  this, SLOT(licenseWindowEvent()) );
   m_helpMenu->insertItem( "About...",    this, SLOT(aboutWindowEvent()) );

   //m_menuBar = new QMenuBar( this );
   m_menuBar = menuBar();
   m_menuBar->insertItem( "&File", m_fileMenu );
   m_menuBar->insertItem( "&View", m_viewMenu );
   m_menuBar->insertItem( "&Tools", m_toolMenu );
   m_menuBar->insertItem( "&Primitives", m_primitiveMenu );
   m_menuBar->insertItem( "&Groups", m_groupMenu );
   m_menuBar->insertItem( "&Animation", m_animMenu );
   m_menuBar->insertItem( "&Help", m_helpMenu );
   //m_menuBar->insertItem( "&Scripts", m_scriptMenu );

   // this is probably unnecessary now
   m_model->setUndoEnabled( true );
   
   resize( 
         g_prefs.exists( "ui_viewwin_width")  ? g_prefs( "ui_viewwin_width" )  : 900,
         g_prefs.exists( "ui_viewwin_height") ? g_prefs( "ui_viewwin_height" ) : 700 );

   setMinimumSize( 520, 520 );

   frameAllEvent();

   m_viewPanel->modelUpdatedEvent();
   show();

   if ( check_update_should_check() )
   {
      QTimer::singleShot( 5000, this, SLOT(updateTimeoutCheck()) );
   }

   QTimer * m_savedTimer = new QTimer();
   connect( m_savedTimer, SIGNAL(timeout()), this, SLOT(savedTimeoutCheck()) );
   m_savedTimer->start( 1000 );
}

ViewWindow::~ViewWindow()
{
   log_debug( "deleting view window\n" );
   _winList.remove( this );

   log_debug( "deleting view window %08X, model %08X\n", this, m_model );
   DecalManager::getInstance()->unregisterModel( m_model );
   m_viewPanel->freeTextures();
   delete m_model;
   m_viewPanel->setModel( NULL );

   model_show_alloc_stats();

   // QToolBar actually deletes buttons, we just need to delete array
   delete[] m_toolButtons;

   if ( !_shuttingDown )
   {
      if ( _winList.empty() )
      {
         ui_exit();
      }
   }

   delete m_toolbox;
}

void ViewWindow::setModel( Model * model )
{
   m_viewPanel->setModel( model );
   m_statusBar->setModel( model );
   m_model = model;

   if ( m_model )
   {
      Model::DrawJointMode jointMode = (Model::DrawJointMode) g_prefs( "ui_draw_joints" ).intValue();

      if ( jointMode != Model::JOINTMODE_LINES )
      {
         jointMode = Model::JOINTMODE_BONES;
      }
      m_model->setDrawJoints( jointMode );

      g_prefs( "ui_draw_joints" ) = (int) jointMode;
   }

   updateCaption();

   m_viewPanel->modelUpdatedEvent();
}

bool ViewWindow::getSaved()
{
   if ( m_model )
   {
      return m_model->getSaved();
   }
   return true;
}

/*
void ViewWindow::resizeEvent ( QResizeEvent * e )
{
   int w = e->size().width();
   int h = e->size().height();

   m_menuBar->move( 0, 0 );
   m_menuBar->resize( w, m_menuBar->height() );

   m_viewPanel->move( 0, m_menuBar->height() );
   m_viewPanel->resize( w, h - m_menuBar->height() - m_statusBar->height() );

   m_statusBar->move( 0, h - m_statusBar->height() );
   m_statusBar->resize( w, m_statusBar->height() );

   g_prefs( "ui_viewwin_width" )  = w;
   g_prefs( "ui_viewwin_height" ) = h;
}
*/

void ViewWindow::helpNowEvent( int id )
{
   HelpWin * win = new HelpWin( "olh_viewwin.html", false );
   win->show();
}

void ViewWindow::saveModelEvent()
{
   const char * filename = m_model->getFilename();
   if ( filename && filename[0] )
   {
      Model::ModelError err 
         = FilterManager::getInstance()->writeFile( m_model, filename );

      if ( err == Model::ERROR_NONE )
      {
         m_model->setSaved( true );
         prefs_recent_model( filename );
      }
      else
      {
         m_abortQuit = true;
         if ( err != Model::ERROR_CANCEL )
         {
            QString reason = Model::errorToString( err, m_model );
            reason = tr(filename) + tr(":\n") + reason;
            msg_error( reason.latin1() );
         }
      }
   }
   else
   {
      saveModelAsEvent();
   }
}

void ViewWindow::saveModelAsEvent()
{
   list<string> formats = FilterManager::getInstance()->getAllWriteTypes();

   QString formatsStr = "All Supported Formats (";

   list<string>::iterator it = formats.begin();
   while(  it != formats.end() )
   {
      formatsStr += QString( (*it).c_str() );

      it++;

      if ( it != formats.end() )
      {
         formatsStr += " ";
      }
   }

   formatsStr += ")";

   const char * modelFile = m_model->getFilename();
   QString dir = g_prefs( "ui_model_dir" );
   if ( modelFile && modelFile[0] != '\0' )
   {
      std::string fullname;
      std::string fullpath;
      std::string basename;

      normalizePath( modelFile, fullname, fullpath, basename );
      dir = tr( fullpath.c_str() );
   }

   if ( dir.isEmpty() )
   {
      dir = ".";
   }

#ifdef HAVE_QT4
   QFileDialog d(NULL, "", dir, formatsStr + QString( ";; All Files (*)" ) );
   d.setAcceptMode( QFileDialog::AcceptSave );
#else
   QFileDialog d(dir, formatsStr, NULL, "", true );
   d.addFilter( "All Files (*)" );
#endif

   d.setCaption( "Save model file as" );
   d.setSelectedFilter( formatsStr );
   d.setMode( QFileDialog::AnyFile );

   m_abortQuit = true;

   bool again = true;
   while ( again )
   {
      again = false;
      if ( QDialog::Accepted == d.exec() )
      {
         bool save = true;
#ifndef HAVE_QT4
         if ( file_exists(d.selectedFile().latin1()) )
         {
            char val = msg_warning_prompt( "File exists.  Overwrite?", "yNc" );
            switch ( val )
            {
               case 'N':
                  again = true;

                  // We want to fall through here

               case 'C':
                  save = false;
                  break;

               default:
                  break;
            }
         }
#endif // HAVE_QT4

         if ( save )
         {
            Model::ModelError err 
               = FilterManager::getInstance()->writeFile( m_model, d.selectedFile().latin1() );
            if ( err == Model::ERROR_NONE )
            {
               m_abortQuit = false;
               m_model->setSaved( true );
               m_model->setFilename( d.selectedFile().latin1() );
               prefs_recent_model( d.selectedFile() );
#ifdef HAVE_QT4
               g_prefs( "ui_model_dir" ) = d.directory().absolutePath();
#else
               g_prefs( "ui_model_dir" ) = d.dir()->absPath();
#endif 

               updateCaption();
            }
            else
            {
               if ( err != Model::ERROR_CANCEL )
               {
                  msg_error( Model::errorToString( err, m_model ) );
               }
            }
         }
      }
   }
}

void ViewWindow::mergeModelsEvent()
{
   list<string> formats = FilterManager::getInstance()->getAllReadTypes();

   QString formatsStr = "All Supported Formats (";

   list<string>::iterator it = formats.begin();
   while(  it != formats.end() )
   {
      formatsStr += QString( (*it).c_str() );

      it++;

      if ( it != formats.end() )
      {
         formatsStr += " ";
      }
   }

   formatsStr += ")";

   QString dir = g_prefs( "ui_model_dir" );
   if ( dir.isEmpty() )
   {
      dir = ".";
   }

#ifdef HAVE_QT4
   QFileDialog d(NULL, "", dir, formatsStr + QString( ";; All Files (*)" ) );
#else
   QFileDialog d(dir, formatsStr, NULL, "", true );
   d.addFilter( "All Files (*)" );
#endif

   d.setCaption( "Open model file" );
   d.setSelectedFilter( formatsStr );

   if ( QDialog::Accepted == d.exec() )
   {
      char * filename = strdup( d.selectedFile().latin1() );

      Model::ModelError err;
      Model * model = new Model();
      if ( (err = FilterManager::getInstance()->readFile( model, filename )) == Model::ERROR_NONE)
      {
         model_show_alloc_stats();

         MergeWindow mw( model, this );

         if ( mw.exec() )
         {
            Model::AnimationMerge mode = Model::AM_NONE;

            if ( mw.getIncludeAnimation() )
            {
               mode = Model::AM_ADD;

               if ( mw.getAnimationMerge() )
               {
                  mode = Model::AM_MERGE;
               }
            }
            double rot[3];
            double trans[3];
            mw.getRotation( rot );
            mw.getTranslation( trans );
            m_model->mergeModels( model, mw.getIncludeTexture(), mode, true, trans, rot );
            m_model->operationComplete( "Merge models" );
#ifdef HAVE_QT4
            g_prefs( "ui_model_dir" ) = d.directory().absolutePath();
#else
            g_prefs( "ui_model_dir" ) = d.dir()->absPath();
#endif 

            m_viewPanel->modelUpdatedEvent();
         }

         prefs_recent_model( filename );
         delete model;
      }
      else
      {
         QString reason = Model::errorToString( err, model );
         reason = tr(filename) + tr(":\n") + reason;
         msg_error( reason.latin1() );
         delete model;
      }

      free( filename );
   }
}

void ViewWindow::mergeAnimationsEvent()
{
   list<string> formats = FilterManager::getInstance()->getAllReadTypes();

   QString formatsStr = "All Supported Formats (";

   list<string>::iterator it = formats.begin();
   while(  it != formats.end() )
   {
      formatsStr += QString( (*it).c_str() );

      it++;

      if ( it != formats.end() )
      {
         formatsStr += " ";
      }
   }

   formatsStr += ")";

   QString dir = g_prefs( "ui_model_dir" );
   if ( dir.isEmpty() )
   {
      dir = ".";
   }

#ifdef HAVE_QT4
   QFileDialog d(NULL, "", dir, formatsStr + QString( ";; All Files (*)" ) );
#else
   QFileDialog d(dir, formatsStr, NULL, "", true );
   d.addFilter( "All Files (*)" );
#endif

   d.setCaption( "Open model file" );
   d.setSelectedFilter( formatsStr );

   if ( QDialog::Accepted == d.exec() )
   {
      char * filename = strdup( d.selectedFile().latin1() );

      Model::ModelError err;
      Model * model = new Model();
      if ( (err = FilterManager::getInstance()->readFile( model, filename )) == Model::ERROR_NONE)
      {
         model_show_alloc_stats();

         m_model->mergeAnimations( model );

         prefs_recent_model( filename );
         delete model;
      }
      else
      {
         QString reason = Model::errorToString( err, model );
         reason = tr(filename) + tr(":\n") + reason;
         msg_error( reason.latin1() );
         delete model;
      }

      free( filename );
   }
}

void ViewWindow::scriptEvent()
{
#ifdef HAVE_LUALIB
   QString dir = g_prefs( "ui_script_dir" );
   if ( dir.isEmpty() )
   {
      dir = ".";
   }

   QString formatsStr = "Lua scripts (*.lua)";
#ifdef HAVE_QT4
   QFileDialog d(NULL, "", dir, formatsStr + QString( ";; All Files (*)" ) );
#else
   QFileDialog d(dir, formatsStr, NULL, "", true );
   d.addFilter( "All Files (*)" );
#endif

   d.setCaption( "Open model file" );
   d.setSelectedFilter( formatsStr );

   if ( QDialog::Accepted == d.exec() )
   {
#ifdef HAVE_QT4
      g_prefs( "ui_script_dir" ) = d.directory().absolutePath();
#else
      g_prefs( "ui_script_dir" ) = d.dir()->absPath();
#endif 

      char * filename = strdup( d.selectedFile().latin1() );
      runScript( filename );

      free( filename );
   }
#endif // HAVE_LUALIB
}

void ViewWindow::runScript( const char * filename )
{
#ifdef HAVE_LUALIB
   if ( filename )
   {
      LuaScript lua;
      LuaContext lc( m_model );
      luaif_registerfunctions( &lua, &lc );

      std::string scriptfile = filename;

      std::string fullname;
      std::string fullpath;
      std::string basename;

      normalizePath( scriptfile.c_str(), fullname, fullpath, basename );

      log_debug( "running script %s\n", basename.c_str() );
      int rval = lua.runFile( scriptfile.c_str() );

      prefs_recent_script( filename );

      if ( rval == 0 )
      {
         log_debug( "script complete, exited normally\n" );
         model_status( m_model, StatusNormal, STATUSTIME_SHORT, "Script %s complete", basename.c_str() );
      }
      else
      {
         log_error( "script complete, exited with error code %d\n", rval );
         model_status( m_model, StatusError, STATUSTIME_LONG, "Script %s error: %s", basename.c_str(), lua.error() );
      }

      m_model->setNoAnimation();
      m_model->operationComplete( basename.c_str() );

      m_viewPanel->modelUpdatedEvent();
   }
#endif // HAVE_LUALIB
}

void ViewWindow::closeEvent( QCloseEvent * e )
{
#ifdef CODE_DEBUG
   e->accept();
#else // CODE_DEBUG
   if ( ! m_model->getSaved() )
   {
      int val = QMessageBox::warning( this, "Save first?", "Model has been modified\nDo you want to save before closing?", QMessageBox::Yes, QMessageBox::No, QMessageBox::Cancel );
      switch ( val )
      {
         case QMessageBox::Yes:
            m_abortQuit = false;
            saveModelEvent();
            if ( ! m_abortQuit )
            {
               e->accept();
            }
            break;
         case QMessageBox::No:
            e->accept();
            break;
         case QMessageBox::Cancel:
            e->ignore();
            break;
         default:
            {
               QString str;
               str.sprintf( "Unknown response: %d, Canceling close request\n", val );
               msg_error( str.latin1() );
            }
            e->ignore();
            break;
      }
   }
   else
   {
      e->accept();
   }
#endif // CODE_DEBUG
}

void ViewWindow::frameAllEvent()
{
   double x1, y1, z1, x2, y2, z2;
   if ( m_model->getBoundingRegion( &x1, &y1, &z1, &x2, &y2, &z2 ) )
   {
      m_viewPanel->frameArea( x1, y1, z1, x2, y2, z2 );
   }
}

void ViewWindow::frameSelectedEvent()
{
   double x1, y1, z1, x2, y2, z2;
   if ( m_model->getSelectedBoundingRegion( &x1, &y1, &z1, &x2, &y2, &z2 ) )
   {
      m_viewPanel->frameArea( x1, y1, z1, x2, y2, z2 );
   }
}

void ViewWindow::renderBadEvent()
{
   g_prefs( "ui_render_bad_textures" ) = (g_prefs( "ui_render_bad_textures" ).intValue() == 0) ? 1 : 0;
   int renderBad = g_prefs( "ui_render_bad_textures" );
   m_viewMenu->changeItem( m_renderBadItem, renderBad ? "Hide Bad Textures" : "Render Bad Textures" );
   m_viewPanel->modelUpdatedEvent();
}

void ViewWindow::renderSelectionEvent()
{
   g_prefs( "ui_render_3d_selections" ) = (g_prefs( "ui_render_3d_selections" ).intValue() == 0) ? 1 : 0;
   int renderSelection = g_prefs( "ui_render_3d_selections" );
   m_viewMenu->changeItem( m_renderSelectionItem, renderSelection ? "Hide 3D Selection" : "Render 3D Selections" );
   m_viewPanel->modelUpdatedEvent();
}

void ViewWindow::toolActivated( int id )
{
   ToolMenuItemList::iterator it;
   it = m_tools.begin();
   while ( it != m_tools.end() )
   {
      if ( (*it)->id == id )
      {
         ::Tool * tool = (*it)->tool;
         for ( int t = 0; t < m_toolCount; t++ )
         {
            if ( m_toolList[t] == tool )
            {
               if ( !m_toolButtons[t]->isOn() )
               {
                  m_toolButtons[t]->setOn( true );
                  //StatusBar::setAllTools( m_toolList[t]->getName(0), QPixmap( *m_toolButtons[t]->pixmap() ) );
               }
               /*
               if ( m_currentTool )
               {
                  m_currentTool->deactivated();
               }
               */
               m_currentTool = m_toolList[ t ];

               //tool->activated( (*it)->arg, m_model, this );
            }
         }
         return;
      }
      it++;
   }
}

void ViewWindow::primitiveCommandActivated( int id )
{
   CommandMenuItemList::iterator it;
   it = m_primitiveCommands.begin();
   while ( it != m_primitiveCommands.end() )
   {
      if ( (*it)->id == id )
      {
         if ( ((*it)->command)->activated( (*it)->arg, m_model ) )
         {
            m_model->operationComplete( ((*it)->command)->getName( (*it)->arg ) );
            m_viewPanel->modelUpdatedEvent();
         }
         break;
      }
      it++;
   }
}

void ViewWindow::groupCommandActivated( int id )
{
   CommandMenuItemList::iterator it;
   it = m_groupCommands.begin();
   while ( it != m_groupCommands.end() )
   {
      if ( (*it)->id == id )
      {
         if ( ((*it)->command)->activated( (*it)->arg, m_model ) )
         {
            m_model->operationComplete( ((*it)->command)->getName( (*it)->arg ) );
            m_viewPanel->modelUpdatedEvent();
         }
         break;
      }
      it++;
   }
}

void ViewWindow::scriptActivated( int id )
{
}

void ViewWindow::groupWindowEvent()
{
   GroupWindow * win = new GroupWindow( m_model );
   win->show();
}

void ViewWindow::textureWindowEvent()
{
   TextureWindow * win = new TextureWindow( m_model );
   win->show();
}

void ViewWindow::textureCoordEvent()
{
   list<int> list = m_model->getSelectedTriangles();

   if ( ! list.empty() )
   {
      TextureCoord * win = new TextureCoord( m_model );
      win->show();
   }
   else
   {
      msg_info( "You must select faces first.\nUse the 'Select' tool and click on 'Faces'." );
   }
}

void ViewWindow::metaWindowEvent()
{
   MetaWindow * win = new MetaWindow( m_model );
   win->show();
}

void ViewWindow::reloadTexturesEvent()
{
   if( TextureManager::getInstance()->reloadTextures() )
   {
      invalidateModelTextures();
   }
}


void ViewWindow::undoRequest()
{
   log_debug( "undo request\n" );

   if ( m_model->canUndo() )
   {
      const char * opname = m_model->getUndoOpName();

      if ( m_animWin )
      {
         m_animWin->undoRequest();
      }
      else
      {
         m_model->undo();

         if ( m_model->getAnimationMode() )
         {
            m_animWin = new AnimWindow( m_model, true, this );
            connect( m_animWin, SIGNAL(animWindowClosed()), this, SLOT(animationModeDone()) );
            m_animMenu->setItemEnabled( m_animExportItem, false );
            m_animMenu->setItemEnabled( m_animSetsItem, false );
            m_animMenu->setItemEnabled( m_startAnimItem, false );
            m_animMenu->setItemEnabled( m_stopAnimItem,  true  );
            m_animMenu->setItemEnabled( m_animSetRotItem,  true  );
            m_animMenu->setItemEnabled( m_animSetTransItem,  true  );
            m_animWin->show();
         }
         else
         {
            m_viewPanel->modelUpdatedEvent();
         }
      }
      
      model_status ( m_model, StatusNormal, STATUSTIME_SHORT, "Undo %s", 
            (opname && opname[0]) ? opname : "" );

      if ( m_model->getSelectedBoneJointCount() > 0 )
      {
         m_model->setDrawJoints( 
               (Model::DrawJointMode) g_prefs( "ui_draw_joints" ).intValue() );
         m_viewPanel->modelUpdatedEvent();
      }
   }
   else
   {
      model_status( m_model, StatusNormal, STATUSTIME_SHORT, "Nothing to undo" );
   }
}

void ViewWindow::redoRequest()
{
   log_debug( "redo request\n" );

   if ( m_model->canRedo() )
   {
      const char * opname = m_model->getRedoOpName();

      if ( m_animWin )
      {
         m_animWin->redoRequest();
      }
      else
      {
         m_model->redo();

         if ( m_model->getAnimationMode() )
         {
            m_animWin = new AnimWindow( m_model, true, this );
            connect( m_animWin, SIGNAL(animWindowClosed()), this, SLOT(animationModeDone()) );
            m_animMenu->setItemEnabled( m_animExportItem, false );
            m_animMenu->setItemEnabled( m_animSetsItem, false );
            m_animMenu->setItemEnabled( m_startAnimItem, false );
            m_animMenu->setItemEnabled( m_stopAnimItem,  true  );
            m_animMenu->setItemEnabled( m_animSetRotItem,  true  );
            m_animMenu->setItemEnabled( m_animSetTransItem,  true  );
            m_animWin->show();
         }
         else
         {
            m_viewPanel->modelUpdatedEvent();
         }
      }

      if ( m_model->getSelectedBoneJointCount() > 0 )
      {
         m_model->setDrawJoints( 
               (Model::DrawJointMode) g_prefs( "ui_draw_joints" ).intValue() );
         m_viewPanel->modelUpdatedEvent();
      }
      
      model_status ( m_model, StatusNormal, STATUSTIME_SHORT, "Redo %s", 
            (opname && opname[0]) ? opname : "" );
   }
   else
   {
      model_status( m_model, StatusNormal, STATUSTIME_SHORT, "Nothing to redo" );
   }
}

void ViewWindow::helpWindowEvent()
{
   HelpWin * win = new HelpWin();
   win->show();
}

void ViewWindow::aboutWindowEvent()
{
   AboutWin * win = new AboutWin();
   win->show();
}

void ViewWindow::licenseWindowEvent()
{
   LicenseWin * win = new LicenseWin();
   win->show();
}

void ViewWindow::animSetWindowEvent()
{
   AnimSetWindow asw( m_model, this );
   asw.exec();
}

void ViewWindow::animExportWindowEvent()
{
   if ( m_model->getAnimCount( Model::ANIMMODE_SKELETAL ) > 0 
         || m_model->getAnimCount( Model::ANIMMODE_FRAME ) > 0 )
   {
      AnimExportWindow aew( m_model, m_viewPanel, this );
      aew.exec();
   }
   else
   {
      msg_error( "This model does not have any animations" );
   }
}

void ViewWindow::animSetRotEvent()
{
   double point[3] = { 0.0, 0.0, 0.0 };
   Matrix m;
   m.loadIdentity();
   m_model->rotateSelectedVerticesOnPoint( m, point );
   m_model->operationComplete( "Set rotation keframe" );
}

void ViewWindow::animSetTransEvent()
{
   Matrix m;
   m.loadIdentity();
   m_model->translateSelectedVertices( m );
   m_model->operationComplete( "Set translation keframe" );
}

void ViewWindow::startAnimationMode()
{
   m_animWin = new AnimWindow( m_model, false, this );
   connect( m_animWin, SIGNAL(animWindowClosed()), this, SLOT(animationModeDone()) );
   m_animMenu->setItemEnabled( m_animExportItem, false );
   m_animMenu->setItemEnabled( m_animSetsItem, false );
   m_animMenu->setItemEnabled( m_startAnimItem, false );
   m_animMenu->setItemEnabled( m_stopAnimItem,  true  );
   m_animMenu->setItemEnabled( m_animSetRotItem,  true  );
   m_animMenu->setItemEnabled( m_animSetTransItem,  true  );
   m_animWin->show();
}

void ViewWindow::stopAnimationMode()
{
   if ( m_animWin )
   {
      m_animWin->close();
   }
}

void ViewWindow::animationModeDone()
{
   m_animWin = NULL;
   m_animMenu->setItemEnabled( m_animExportItem, true );
   m_animMenu->setItemEnabled( m_animSetsItem, true );
   m_animMenu->setItemEnabled( m_startAnimItem, true );
   m_animMenu->setItemEnabled( m_stopAnimItem,  false  );
   m_animMenu->setItemEnabled( m_animSetRotItem,  false  );
   m_animMenu->setItemEnabled( m_animSetTransItem,  false  );
}


void ViewWindow::buttonToggled( bool on )
{
   QToolButton * newDown = NULL;
   int newDownIndex = 0;

   int downCount = 0;

   for ( int t = 0; t < m_toolCount; t++ )
   {
      if ( m_toolButtons[t]->isOn() )
      {
         downCount++;
         if ( m_last != m_toolButtons[t] )
         {
            newDown = m_toolButtons[t];
            newDownIndex = t;
         }
      }
   }

   if ( m_last && downCount > 1 )
   {
      m_last->setOn( false );
   }

   if ( newDown )
   {
      if ( on )
      {
         if ( m_currentTool )
         {
            m_currentTool->deactivated();
         }
         m_currentTool = m_toolList[ newDownIndex ];
         m_currentTool->activated( 0, m_model, this );

         m_last = newDown;
         m_toolbox->setCurrentTool( m_currentTool );
         //changeSelectedToolWidget( m_currentTool );
         //StatusBar::setAllTools( m_currentTool->getName(0), QPixmap( *m_last->pixmap() ) );
      }
   }
   else
   {
      if ( m_last )
      {
         m_last->setOn( true );
      }
   }
}

void ViewWindow::initializeToolbox()
{
   m_toolbox->registerAllTools();

   m_toolButtons = new QToolButtonPtr[ m_toolbox->getToolCount() ];
   m_toolList    = new ToolPtr[ m_toolbox->getToolCount() ];
   m_toolCount = 0;

   ::Tool * tool = m_toolbox->getFirstTool();
   while ( tool )
   {
      if ( tool->isManipulation() )
      {
         ToolMenuItem * item;
         int id;
         int keyBinding = 0;
         int count = tool->getToolCount();
         if ( count > 1 )
         {
            QPopupMenu * menu = new QPopupMenu( this );
            for ( int t = 1; t < count; t++ )
            {
               keyBinding = 0;
               const char * name = tool->getName( t );
               id = menu->insertItem( name );

               if ( tool->getKeyBinding( t, keyBinding ) && keyBinding )
               {
                  menu->setAccel( keyBinding, id );
               }
               item = new ToolMenuItem;
               item->id = id;
               item->tool = tool;
               item->arg = t;

               m_tools.push_back( item );

               // Create tool button
               QIconSet set;
#ifdef HAVE_QT4
               set.setPixmap( QPixmap( tool->getPixmap() ), QIconSet::Small );
#else
               set.setPixmap( QPixmap( tool->getPixmap() ), SmallIconSize );
#endif
               m_toolList[m_toolCount] = tool;
               m_toolButtons[ m_toolCount ] = new QToolButton( m_toolBar );
               m_toolButtons[ m_toolCount ]->setToggleButton( true );
               m_toolButtons[ m_toolCount ]->setIconSet( set );
               if ( name && name[0] )
               {
                  QToolTip::add( m_toolButtons[ m_toolCount ] , name );
               }

               connect( m_toolButtons[m_toolCount], SIGNAL(toggled(bool)), this, SLOT(buttonToggled(bool)));

               m_toolCount++;
            }

            id = m_toolMenu->insertItem( tool->getName(0), menu );
            keyBinding = 0;
            if ( tool->getKeyBinding( 0, keyBinding ) && keyBinding )
            {
               menu->setAccel( keyBinding, id );
            }
            item = new ToolMenuItem;
            item->id = id;
            item->tool = tool;
            item->arg = 0;

            m_tools.push_back( item );
            connect( menu, SIGNAL(activated(int)), this, SLOT(toolActivated(int)));
         }
         else
         {
            const char * name = tool->getName( 0 );
            id = m_toolMenu->insertItem( name );
            keyBinding = 0;
            if ( tool->getKeyBinding( 0, keyBinding ) && keyBinding )
            {
               m_toolMenu->setAccel( keyBinding, id );
            }
            item = new ToolMenuItem;
            item->id = id;
            item->tool = tool;
            item->arg = 0;

            m_tools.push_back( item );

            // Create tool button
            QIconSet set;
#ifdef HAVE_QT4
            set.setPixmap( QPixmap( tool->getPixmap() ), QIconSet::Small );
#else
            set.setPixmap( QPixmap( tool->getPixmap() ), SmallIconSize );
#endif
            m_toolList[m_toolCount] = tool;
            m_toolButtons[ m_toolCount ] = new QToolButton( m_toolBar );
            m_toolButtons[ m_toolCount ]->setToggleButton( true );
            m_toolButtons[ m_toolCount ]->setIconSet( set );
            if ( name && name[0] )
            {
               QToolTip::add( m_toolButtons[ m_toolCount ] , name );
            }

            connect( m_toolButtons[m_toolCount], SIGNAL(toggled(bool)), this, SLOT(buttonToggled(bool)));

            m_toolCount++;
         }
      }
      tool = m_toolbox->getNextTool();
   }
   m_toolMenu->insertSeparator();
   tool = m_toolbox->getFirstTool();
   while ( tool )
   {
      if ( tool->isCreation() )
      {
         ToolMenuItem * item;
         int id;
         int keyBinding = 0;
         int count = tool->getToolCount();
         if ( count > 1 )
         {
            QPopupMenu * menu = new QPopupMenu( this );
            for ( int t = 1; t < count; t++ )
            {
               const char * name = tool->getName( t );
               keyBinding = 0;
               id = menu->insertItem( name );

               if ( tool->getKeyBinding( t, keyBinding ) && keyBinding )
               {
                  menu->setAccel( keyBinding, id );
               }
               item = new ToolMenuItem;
               item->id = id;
               item->tool = tool;
               item->arg = t;

               m_tools.push_back( item );

               // Create tool button
               QIconSet set;
#ifdef HAVE_QT4
               set.setPixmap( QPixmap( tool->getPixmap() ), QIconSet::Small );
#else
               set.setPixmap( QPixmap( tool->getPixmap() ), SmallIconSize );
#endif
               m_toolList[m_toolCount] = tool;
               m_toolButtons[ m_toolCount ] = new QToolButton( m_toolBar );
               m_toolButtons[ m_toolCount ]->setToggleButton( true );
               m_toolButtons[ m_toolCount ]->setIconSet( set );
               if ( name && name[0] )
               {
                  QToolTip::add( m_toolButtons[ m_toolCount ] , name );
               }

               connect( m_toolButtons[m_toolCount], SIGNAL(toggled(bool)), this, SLOT(buttonToggled(bool)));

               m_toolCount++;
            }

            id = m_toolMenu->insertItem( tool->getName(0), menu );
            keyBinding = 0;
            if ( tool->getKeyBinding( 0, keyBinding ) && keyBinding )
            {
               menu->setAccel( keyBinding, id );
            }
            item = new ToolMenuItem;
            item->id = id;
            item->tool = tool;
            item->arg = 0;

            m_tools.push_back( item );
            connect( menu, SIGNAL(activated(int)), this, SLOT(toolActivated(int)));
         }
         else
         {
            const char * name = tool->getName( 0 );
            id = m_toolMenu->insertItem( name );
            keyBinding = 0;
            if ( tool->getKeyBinding( 0, keyBinding ) && keyBinding )
            {
               m_toolMenu->setAccel( keyBinding, id );
            }
            item = new ToolMenuItem;
            item->id = id;
            item->tool = tool;
            item->arg = 0;

            m_tools.push_back( item );

            // Create tool button
            QIconSet set;
#ifdef HAVE_QT4
            set.setPixmap( QPixmap( tool->getPixmap() ), QIconSet::Small );
#else
            set.setPixmap( QPixmap( tool->getPixmap() ), SmallIconSize );
#endif
            m_toolList[m_toolCount] = tool;
            m_toolButtons[ m_toolCount ] = new QToolButton( m_toolBar );
            m_toolButtons[ m_toolCount ]->setToggleButton( true );
            m_toolButtons[ m_toolCount ]->setIconSet( set );
            if ( name && name[0] )
            {
               QToolTip::add( m_toolButtons[ m_toolCount ] , name );
            }

            connect( m_toolButtons[m_toolCount], SIGNAL(toggled(bool)), this, SLOT(buttonToggled(bool)));

            m_toolCount++;
         }
      }
      tool = m_toolbox->getNextTool();
   }
   connect( m_toolMenu, SIGNAL(activated(int)), this, SLOT(toolActivated(int)));

}

void ViewWindow::initializeCommands()
{
   //m_cmdMgr = new CommandManager();
   //init_std_cmds( m_cmdMgr );
   m_cmdMgr = CommandManager::getInstance();
   Command * cmd = m_cmdMgr->getFirstCommand();
   while ( cmd )
   {
      CommandMenuItem * item;
      int id;
      int keyBinding = 0;
      int count = cmd->getCommandCount();
      if ( count > 1 )
      {
         QPopupMenu * menu = new QPopupMenu( this );
         for ( int t = 1; t < count; t++ )
         {
            keyBinding = 0;
            id = menu->insertItem( cmd->getName(t) );

            if ( cmd->getKeyBinding( t, keyBinding ) && keyBinding )
            {
               menu->setAccel( keyBinding, id );
            }
            item = new CommandMenuItem;
            item->id = id;
            item->command = cmd;
            item->arg = t;

            if ( cmd->isPrimitive() )
            {
               m_primitiveCommands.push_back( item );
            }
            else
            {
               m_groupCommands.push_back( item );
            }
         }

         log_debug( "adding command '%s' to menus\n", cmd->getName(0) );
         if ( cmd->isPrimitive() )
         {
            id = m_primitiveMenu->insertItem( cmd->getName(0), menu );
#ifndef HAVE_QT4
            connect( menu, SIGNAL(activated(int)), this, SLOT(primitiveCommandActivated(int)));
#endif 
         }
         else
         {
            id = m_groupMenu->insertItem( cmd->getName(0), menu );
#ifndef HAVE_QT4
            connect( menu, SIGNAL(activated(int)), this, SLOT(groupCommandActivated(int)));
#endif 
         }

         keyBinding = 0;
         if ( cmd->getKeyBinding( 0, keyBinding ) && keyBinding )
         {
            menu->setAccel( keyBinding, id );
         }
         item = new CommandMenuItem;
         item->id = id;
         item->command = cmd;
         item->arg = 0;

         if ( cmd->isPrimitive() )
         {
            m_primitiveCommands.push_back( item );
         }
         else
         {
            m_groupCommands.push_back( item );
         }
      }
      else
      {
         QPopupMenu * curMenu = cmd->isPrimitive() ? m_primitiveMenu : m_groupMenu;
         id = curMenu->insertItem( cmd->getName(0) );
         keyBinding = 0;
         if ( cmd->getKeyBinding( 0, keyBinding ) && keyBinding )
         {
            curMenu->setAccel( keyBinding, id );
         }
         item = new CommandMenuItem;
         item->id = id;
         item->command = cmd;
         item->arg = 0;

         log_debug( "adding command '%s' to menus\n", cmd->getName(0) );
         if ( cmd->isPrimitive() )
         {
            m_primitiveCommands.push_back( item );
         }
         else
         {
            m_groupCommands.push_back( item );
         }
      }
      cmd = m_cmdMgr->getNextCommand();
   }
}

void ViewWindow::fillMruMenu()
{
   m_mruMenu->clear();
   for ( unsigned i = 0; i < g_prefs("mru").count(); i++ )
   {
      m_mruMenu->insertItem( g_prefs("mru")[i] );
   }
}

void ViewWindow::openMru( int id )
{
   //log_debug( "open %s\n", g_prefs("mru")[id].stringValue().latin1() );
   openModelInWindow( m_mruMenu->text(id).latin1() );
}

void ViewWindow::fillScriptMruMenu()
{
   m_scriptMruMenu->clear();
   for ( unsigned i = 0; i < g_prefs("script_mru").count(); i++ )
   {
      m_scriptMruMenu->insertItem( g_prefs("script_mru")[i] );
   }
}

void ViewWindow::openScriptMru( int id )
{
   //log_debug( "open %s\n", g_prefs("mru")[id].stringValue().latin1() );
   runScript( m_scriptMruMenu->text(id).latin1() );
}

void ViewWindow::openModelEvent()
{
   openModelDialogInWindow();
}

bool ViewWindow::openModel( const char * filename )
{
   bool opened = false;

   log_debug( " file: %s\n", filename );

   Model::ModelError err;
   Model * model = new Model();
   if ( (err = FilterManager::getInstance()->readFile( model, filename )) == Model::ERROR_NONE)
   {
      opened = true;

      model_show_alloc_stats();

      model->setSaved( true );
      ViewWindow * win = new ViewWindow( model, NULL, "" );
      win->getSaved(); // Just so I don't have a warning

      prefs_recent_model( filename );
   }
   else
   {
      QString reason = Model::errorToString( err, model );
      reason = tr(filename) + tr(":\n") + reason;
      msg_error( reason.latin1() );
      delete model;
   }

   return opened;
}

bool ViewWindow::openModelDialog( const char * openDirectory )
{
   bool opened = false;

   list<string> formats = FilterManager::getInstance()->getAllReadTypes();

   QString formatsStr = "All Supported Formats (";

   list<string>::iterator it = formats.begin();
   while(  it != formats.end() )
   {
      formatsStr += QString( (*it).c_str() );

      it++;

      if ( it != formats.end() )
      {
         formatsStr += " ";
      }
   }

   formatsStr += ")";

   QString dir = g_prefs( "ui_model_dir" );
   if ( dir.isEmpty() )
   {
      dir = ".";
   }

   if ( openDirectory )
   {
      dir = openDirectory;
   }

#ifdef HAVE_QT4
   QFileDialog d(NULL, "", dir, formatsStr + QString( ";; All Files (*)" ) );
#else
   QFileDialog d(dir, formatsStr, NULL, "", true );
   d.addFilter( "All Files (*)" );
#endif

   d.setCaption( "Open model file" );
   d.setSelectedFilter( formatsStr );

   if ( QDialog::Accepted == d.exec() )
   {
      if ( openModel( d.selectedFile().latin1() ) )
      {
         opened = true;
#ifdef HAVE_QT4
         g_prefs( "ui_model_dir" ) = d.directory().absolutePath();
#else
         g_prefs( "ui_model_dir" ) = d.dir()->absPath();
#endif 
      }
   }

   return opened;
}

bool ViewWindow::openModelInWindow( const char * filename )
{
   bool opened = false;

   log_debug( " file: %s\n", filename );

#ifndef CODE_DEBUG
   if ( ! m_model->getSaved() )
   {
      int val = QMessageBox::warning( this, "Save first?", "Model has been modified\nDo you want to save before closing?", QMessageBox::Yes, QMessageBox::No, QMessageBox::Cancel );
      switch ( val )
      {
         case QMessageBox::Yes:
            m_abortQuit = false;
            saveModelEvent();
            if ( m_abortQuit )
            {
               return false;
            }
            break;
         case QMessageBox::No:
            break;
         case QMessageBox::Cancel:
            return false;
            break;
         default:
            {
               QString str;
               str.sprintf( "Unknown response: %d, canceling operation\n", val );
               msg_error( str.latin1() );
            }
            return false;
      }
   }
#endif // CODE_DEBUG

   Model::ModelError err;
   Model * model = new Model();
   if ( (err = FilterManager::getInstance()->readFile( model, filename )) == Model::ERROR_NONE)
   {
      opened = true;

      model_show_alloc_stats();

      Model * oldModel = m_model;
      setModel( model );
      delete oldModel;

      frameAllEvent();

      prefs_recent_model( filename );
   }
   else
   {
      QString reason = Model::errorToString( err, model );
      reason = tr(filename) + tr(":\n") + reason;
      msg_error( reason.latin1() );
      delete model;
   }

   return opened;
}

bool ViewWindow::openModelDialogInWindow( const char * openDirectory )
{
   bool opened = false;

   list<string> formats = FilterManager::getInstance()->getAllReadTypes();

   QString formatsStr = "All Supported Formats (";

   list<string>::iterator it = formats.begin();
   while(  it != formats.end() )
   {
      formatsStr += QString( (*it).c_str() );

      it++;

      if ( it != formats.end() )
      {
         formatsStr += " ";
      }
   }

   formatsStr += ")";

   QString dir = g_prefs( "ui_model_dir" );
   if ( dir.isEmpty() )
   {
      dir = ".";
   }

   if ( openDirectory )
   {
      dir = openDirectory;
   }

#ifdef HAVE_QT4
   QFileDialog d(NULL, "", dir, formatsStr + QString( ";; All Files (*)" ) );
#else
   QFileDialog d(dir, formatsStr, NULL, "", true );
   d.addFilter( "All Files (*)" );
#endif

   d.setCaption( "Open model file" );
   d.setSelectedFilter( formatsStr );

   if ( QDialog::Accepted == d.exec() )
   {
      if ( openModelInWindow( d.selectedFile().latin1() ) )
      {
         opened = true;
#ifdef HAVE_QT4
         g_prefs( "ui_model_dir" ) = d.directory().absolutePath();
#else
         g_prefs( "ui_model_dir" ) = d.dir()->absPath();
#endif 
      }
   }

   return opened;
}

void ViewWindow::invalidateModelTextures()
{
   ViewWindowList::iterator windowIter;
   
   Model * model;
   DecalManager * mgr = DecalManager::getInstance();

   for( windowIter = _winList.begin(); windowIter != _winList.end(); windowIter++ )
   {
      model = (*windowIter)->getModel();
      model->invalidateTextures();
      mgr->modelUpdated( model );
   }
}

void ViewWindow::quitEvent()
{
   if ( ViewWindow::closeAllWindows() )
   {
      qApp->quit();
   }
}

void ViewWindow::pluginWindowEvent()
{
   // pluginWin will delete itself view WDestructiveClose
   PluginWindow * pluginWin = new PluginWindow();
   pluginWin->show();
}

void ViewWindow::backgroundWindowEvent()
{
   // pluginWin will delete itself view WDestructiveClose
   BackgroundWin * win = new BackgroundWin( m_model );  
   win->show();
}

void ViewWindow::newModelEvent()
{
   ViewWindow * win = new ViewWindow( new Model, NULL, "" );
   win->getSaved(); // Just so I don't have a warning
}

void ViewWindow::updateTimeoutCheck()
{
   if ( check_update_is_done() )
   {
      _versionNotify( m_model );
   }
   else
   {
      if ( check_update_should_check() )
      {
         QTimer::singleShot( 1000, this, SLOT(updateTimeoutCheck()) );
      }
   }
}

void ViewWindow::savedTimeoutCheck()
{
   updateCaption();
}

void ViewWindow::updateCaption()
{
   QString caption = "Misfit Model 3D: ";
   if ( m_model )
   {
      caption += m_model->getSaved() ? "" : "* ";

      const char * filename = m_model->getFilename();
      if ( filename && filename[0] )
      {
         std::string fullName;
         std::string fullPath;
         std::string baseName;
         normalizePath( filename, fullName, fullPath, baseName );

         caption += baseName.c_str();
      }
      else
      {
         caption += "[unnamed]";
      }
   }

   setCaption( caption );
}

