/*
    QShed <http://www.icculus.org/qshed>

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

#include "global.h"
#include "image.h"
#include "v_font.h"

#include <stdarg.h>
#include <stdio.h>

#ifdef WIN32
# include <windows.h>
#endif
#include <GL/gl.h>

/* 8x12 font, each row is one character */
static const unsigned char font_chars[128][12] =
{
	{0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00}, /* 0 - null*/
	{0x00,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0x00,0x00}, /* 1 - block cursor */
	{0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00}, /* 2 */
	{0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00}, /* 3 */
	{0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00}, /* 4 */
	{0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00}, /* 5 */
	{0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00}, /* 6 */
	{0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00}, /* 7 - bell \a */
	{0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00}, /* 8 - backspace \b */
	{0x00,0x00,0x00,0x00,0x00,0x12,0x24,0x48,0x24,0x12,0x00,0x00}, /* 9 - tab \t */
	{0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00}, /* 10 - line feed \n */
	{0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00}, /* 11 - vertical tab \v */
	{0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00}, /* 12 - form feed \f */
	{0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00}, /* 13 - carriage return \r */
	{0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00}, /* 14 */
	{0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00}, /* 15 */
	{0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00}, /* 16 */
	{0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00}, /* 17 */
	{0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00}, /* 18 */
	{0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00}, /* 19 */
	{0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00}, /* 20 */
	{0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00}, /* 21 */
	{0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00}, /* 22 */
	{0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00}, /* 23 */
	{0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00}, /* 24 */
	{0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00}, /* 25 */
	{0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00}, /* 26 */
	{0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00}, /* 27 */
	{0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00}, /* 28 */
	{0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00}, /* 29 */
	{0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00}, /* 30 */
	{0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00}, /* 31 */
	{0x00,0x00,0x00,0x00,0x00,0x00,0x08,0x00,0x00,0x00,0x00,0x00}, /* 32 - space */
	{0x00,0x0c,0x1e,0x1e,0x1e,0x0c,0x0c,0x00,0x0c,0x0c,0x00,0x00},
	{0x00,0x66,0x66,0x66,0x24,0x00,0x00,0x00,0x00,0x00,0x00,0x00},
	{0x00,0x36,0x36,0x7f,0x36,0x36,0x36,0x7f,0x36,0x36,0x00,0x00},
	{0x0c,0x0c,0x3e,0x03,0x03,0x1e,0x30,0x30,0x1f,0x0c,0x0c,0x00},
	{0x00,0x00,0x00,0x23,0x33,0x18,0x0c,0x06,0x33,0x31,0x00,0x00},
	{0x00,0x0e,0x1b,0x1b,0x0e,0x5f,0x7b,0x33,0x3b,0x6e,0x00,0x00},
	{0x00,0x0c,0x0c,0x0c,0x06,0x00,0x00,0x00,0x00,0x00,0x00,0x00},
	{0x00,0x30,0x18,0x0c,0x06,0x06,0x06,0x0c,0x18,0x30,0x00,0x00},
	{0x00,0x06,0x0c,0x18,0x30,0x30,0x30,0x18,0x0c,0x06,0x00,0x00},
	{0x00,0x00,0x00,0x66,0x3c,0xff,0x3c,0x66,0x00,0x00,0x00,0x00},
	{0x00,0x00,0x00,0x18,0x18,0x7e,0x18,0x18,0x00,0x00,0x00,0x00},
	{0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x1c,0x1c,0x06,0x00},
	{0x00,0x00,0x00,0x00,0x00,0x7f,0x00,0x00,0x00,0x00,0x00,0x00},
	{0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x1c,0x1c,0x00,0x00},
	{0x00,0x00,0x40,0x60,0x30,0x18,0x0c,0x06,0x03,0x01,0x00,0x00},
	{0x00,0x3e,0x63,0x73,0x7b,0x6b,0x6f,0x67,0x63,0x3e,0x00,0x00},
	{0x00,0x08,0x0c,0x0f,0x0c,0x0c,0x0c,0x0c,0x0c,0x3f,0x00,0x00},
	{0x00,0x1e,0x33,0x33,0x30,0x18,0x0c,0x06,0x33,0x3f,0x00,0x00},
	{0x00,0x1e,0x33,0x30,0x30,0x1c,0x30,0x30,0x33,0x1e,0x00,0x00},
	{0x00,0x30,0x38,0x3c,0x36,0x33,0x7f,0x30,0x30,0x78,0x00,0x00},
	{0x00,0x3f,0x03,0x03,0x03,0x1f,0x30,0x30,0x33,0x1e,0x00,0x00},
	{0x00,0x1c,0x06,0x03,0x03,0x1f,0x33,0x33,0x33,0x1e,0x00,0x00},
	{0x00,0x7f,0x63,0x63,0x60,0x30,0x18,0x0c,0x0c,0x0c,0x00,0x00},
	{0x00,0x1e,0x33,0x33,0x37,0x1e,0x3b,0x33,0x33,0x1e,0x00,0x00},
	{0x00,0x1e,0x33,0x33,0x33,0x3e,0x18,0x18,0x0c,0x0e,0x00,0x00},
	{0x00,0x00,0x00,0x1c,0x1c,0x00,0x00,0x1c,0x1c,0x00,0x00,0x00},
	{0x00,0x00,0x00,0x1c,0x1c,0x00,0x00,0x1c,0x1c,0x18,0x0c,0x00},
	{0x00,0x30,0x18,0x0c,0x06,0x03,0x06,0x0c,0x18,0x30,0x00,0x00},
	{0x00,0x00,0x00,0x00,0x7e,0x00,0x7e,0x00,0x00,0x00,0x00,0x00},
	{0x00,0x06,0x0c,0x18,0x30,0x60,0x30,0x18,0x0c,0x06,0x00,0x00},
	{0x00,0x1e,0x33,0x30,0x18,0x0c,0x0c,0x00,0x0c,0x0c,0x00,0x00},
	{0x00,0x3e,0x63,0x63,0x7b,0x7b,0x7b,0x03,0x03,0x3e,0x00,0x00},
	{0x00,0x0c,0x1e,0x33,0x33,0x33,0x3f,0x33,0x33,0x33,0x00,0x00},
	{0x00,0x3f,0x66,0x66,0x66,0x3e,0x66,0x66,0x66,0x3f,0x00,0x00},
	{0x00,0x3c,0x66,0x63,0x03,0x03,0x03,0x63,0x66,0x3c,0x00,0x00},
	{0x00,0x1f,0x36,0x66,0x66,0x66,0x66,0x66,0x36,0x1f,0x00,0x00},
	{0x00,0x7f,0x46,0x06,0x26,0x3e,0x26,0x06,0x46,0x7f,0x00,0x00},
	{0x00,0x7f,0x66,0x46,0x26,0x3e,0x26,0x06,0x06,0x0f,0x00,0x00},
	{0x00,0x3c,0x66,0x63,0x03,0x03,0x73,0x63,0x66,0x7c,0x00,0x00},
	{0x00,0x33,0x33,0x33,0x33,0x3f,0x33,0x33,0x33,0x33,0x00,0x00},
	{0x00,0x1e,0x0c,0x0c,0x0c,0x0c,0x0c,0x0c,0x0c,0x1e,0x00,0x00},
	{0x00,0x78,0x30,0x30,0x30,0x30,0x33,0x33,0x33,0x1e,0x00,0x00},
	{0x00,0x67,0x66,0x36,0x36,0x1e,0x36,0x36,0x66,0x67,0x00,0x00},
	{0x00,0x0f,0x06,0x06,0x06,0x06,0x46,0x66,0x66,0x7f,0x00,0x00},
	{0x00,0x63,0x77,0x7f,0x7f,0x6b,0x63,0x63,0x63,0x63,0x00,0x00},
	{0x00,0x63,0x63,0x67,0x6f,0x7f,0x7b,0x73,0x63,0x63,0x00,0x00},
	{0x00,0x1c,0x36,0x63,0x63,0x63,0x63,0x63,0x36,0x1c,0x00,0x00},
	{0x00,0x3f,0x66,0x66,0x66,0x3e,0x06,0x06,0x06,0x0f,0x00,0x00},
	{0x00,0x1c,0x36,0x63,0x63,0x63,0x73,0x7b,0x3e,0x30,0x78,0x00},
	{0x00,0x3f,0x66,0x66,0x66,0x3e,0x36,0x66,0x66,0x67,0x00,0x00},
	{0x00,0x1e,0x33,0x33,0x03,0x0e,0x18,0x33,0x33,0x1e,0x00,0x00},
	{0x00,0x3f,0x2d,0x0c,0x0c,0x0c,0x0c,0x0c,0x0c,0x1e,0x00,0x00},
	{0x00,0x33,0x33,0x33,0x33,0x33,0x33,0x33,0x33,0x1e,0x00,0x00},
	{0x00,0x33,0x33,0x33,0x33,0x33,0x33,0x33,0x1e,0x0c,0x00,0x00},
	{0x00,0x63,0x63,0x63,0x63,0x6b,0x6b,0x36,0x36,0x36,0x00,0x00},
	{0x00,0x33,0x33,0x33,0x1e,0x0c,0x1e,0x33,0x33,0x33,0x00,0x00},
	{0x00,0x33,0x33,0x33,0x33,0x1e,0x0c,0x0c,0x0c,0x1e,0x00,0x00},
	{0x00,0x7f,0x73,0x19,0x18,0x0c,0x06,0x46,0x63,0x7f,0x00,0x00},
	{0x00,0x3c,0x0c,0x0c,0x0c,0x0c,0x0c,0x0c,0x0c,0x3c,0x00,0x00},
	{0x00,0x00,0x01,0x03,0x06,0x0c,0x18,0x30,0x60,0x40,0x00,0x00},
	{0x00,0x3c,0x30,0x30,0x30,0x30,0x30,0x30,0x30,0x3c,0x00,0x00},
	{0x08,0x1c,0x36,0x63,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00},
	{0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0xff,0x00},
	{0x0c,0x0c,0x18,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00},
	{0x00,0x00,0x00,0x00,0x1e,0x30,0x3e,0x33,0x33,0x6e,0x00,0x00},
	{0x00,0x07,0x06,0x06,0x3e,0x66,0x66,0x66,0x66,0x3b,0x00,0x00},
	{0x00,0x00,0x00,0x00,0x1e,0x33,0x03,0x03,0x33,0x1e,0x00,0x00},
	{0x00,0x38,0x30,0x30,0x3e,0x33,0x33,0x33,0x33,0x6e,0x00,0x00},
	{0x00,0x00,0x00,0x00,0x1e,0x33,0x3f,0x03,0x33,0x1e,0x00,0x00},
	{0x00,0x1c,0x36,0x06,0x06,0x1f,0x06,0x06,0x06,0x0f,0x00,0x00},
	{0x00,0x00,0x00,0x00,0x6e,0x33,0x33,0x33,0x3e,0x30,0x33,0x1e},
	{0x00,0x07,0x06,0x06,0x36,0x6e,0x66,0x66,0x66,0x67,0x00,0x00},
	{0x00,0x18,0x18,0x00,0x1e,0x18,0x18,0x18,0x18,0x7e,0x00,0x00},
	{0x00,0x30,0x30,0x00,0x3c,0x30,0x30,0x30,0x30,0x33,0x33,0x1e},
	{0x00,0x07,0x06,0x06,0x66,0x36,0x1e,0x36,0x66,0x67,0x00,0x00},
	{0x00,0x1e,0x18,0x18,0x18,0x18,0x18,0x18,0x18,0x7e,0x00,0x00},
	{0x00,0x00,0x00,0x00,0x3f,0x6b,0x6b,0x6b,0x6b,0x63,0x00,0x00},
	{0x00,0x00,0x00,0x00,0x1f,0x33,0x33,0x33,0x33,0x33,0x00,0x00},
	{0x00,0x00,0x00,0x00,0x1e,0x33,0x33,0x33,0x33,0x1e,0x00,0x00},
	{0x00,0x00,0x00,0x00,0x3b,0x66,0x66,0x66,0x66,0x3e,0x06,0x0f},
	{0x00,0x00,0x00,0x00,0x6e,0x33,0x33,0x33,0x33,0x3e,0x30,0x78},
	{0x00,0x00,0x00,0x00,0x37,0x76,0x6e,0x06,0x06,0x0f,0x00,0x00},
	{0x00,0x00,0x00,0x00,0x1e,0x33,0x06,0x18,0x33,0x1e,0x00,0x00},
	{0x00,0x00,0x04,0x06,0x3f,0x06,0x06,0x06,0x36,0x1c,0x00,0x00},
	{0x00,0x00,0x00,0x00,0x33,0x33,0x33,0x33,0x33,0x6e,0x00,0x00},
	{0x00,0x00,0x00,0x00,0x33,0x33,0x33,0x33,0x1e,0x0c,0x00,0x00},
	{0x00,0x00,0x00,0x00,0x63,0x63,0x6b,0x6b,0x36,0x36,0x00,0x00},
	{0x00,0x00,0x00,0x00,0x63,0x36,0x1c,0x1c,0x36,0x63,0x00,0x00},
	{0x00,0x00,0x00,0x00,0x66,0x66,0x66,0x66,0x3c,0x30,0x18,0x0f},
	{0x00,0x00,0x00,0x00,0x3f,0x31,0x18,0x06,0x23,0x3f,0x00,0x00},
	{0x00,0x38,0x0c,0x0c,0x06,0x03,0x06,0x0c,0x0c,0x38,0x00,0x00},
	{0x00,0x18,0x18,0x18,0x18,0x00,0x18,0x18,0x18,0x18,0x00,0x00},
	{0x00,0x07,0x0c,0x0c,0x18,0x30,0x18,0x0c,0x0c,0x07,0x00,0x00},
	{0x00,0xce,0xdb,0x73,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00},
	{0x00,0x00,0x00,0x18,0x0c,0x7e,0x0c,0x18,0x00,0x00,0x00,0x00}, /* 127 - delete */
};

static GLuint font_texture = 0;
static float font_texcoords[128][8];

static const int font_charwidth = 8;
static const int font_charheight = 12;

void loadfont(void)
{
	image_rgba_t *image;
	unsigned char *pixels;
	int x, y, i;

/* generate the image for this font - since it's only 128x96, pad it to 128x128 */
	image = image_alloc(mem_globalpool, 128, 128);
	pixels = image->pixels;

	for (y = 0; y < 128; y++)
	for (x = 0; x < 128; x++, pixels += 4)
	{
		pixels[0] = 0xff;
		pixels[1] = 0xff;
		pixels[2] = 0xff;
		pixels[3] = y<96 && (font_chars[y/12*16+x/8][y%12] & (1<<(x%8))) ? 0xff : 0;
	}

/* generate texcoords, taking into account that the image was padded from 128x96 to 128x128 */
	for (i = 0; i < 128; i++)
	{
#define vf (128.0f / 12.0f) /* 10.666... */
		float left   = (float)(i & 15) * (1.0f / 16.0f);
		float top    = (float)(i >> 4) * (1.0f / vf);
		float right  = left + (1.0f / 16.0f);
		float bottom = top + (1.0f / vf);
#undef vf

		font_texcoords[i][0] = left;
		font_texcoords[i][1] = top;
		font_texcoords[i][2] = right;
		font_texcoords[i][3] = top;
		font_texcoords[i][4] = right;
		font_texcoords[i][5] = bottom;
		font_texcoords[i][6] = left;
		font_texcoords[i][7] = bottom;
	}

/* upload the font image */
	glGenTextures(1, &font_texture);
	glBindTexture(GL_TEXTURE_2D, font_texture);
	glTexImage2D(GL_TEXTURE_2D, 0, 4, image->width, image->height, 0, GL_RGBA, GL_UNSIGNED_BYTE, image->pixels);
	glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_NEAREST);
	glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_NEAREST);
	glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_S, GL_REPEAT);
	glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_T, GL_REPEAT);

/* finished */
	image_free(&image);
}

void freefont(void)
{
	glDeleteTextures(1, &font_texture);
	font_texture = 0;
}

void drawstring(int x, int y, const char *s)
{
	glEnable(GL_BLEND);
	glBlendFunc(GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);
	glEnable(GL_TEXTURE_2D);
	glBindTexture(GL_TEXTURE_2D, font_texture);

	glBegin(GL_QUADS);
	for (; *s; s++, x += font_charwidth)
	{
		if (*s > 32)
		{
			glTexCoord2fv(font_texcoords[(int)*s]+0); glVertex2i(x                 , y                  );
			glTexCoord2fv(font_texcoords[(int)*s]+2); glVertex2i(x + font_charwidth, y                  );
			glTexCoord2fv(font_texcoords[(int)*s]+4); glVertex2i(x + font_charwidth, y + font_charheight);
			glTexCoord2fv(font_texcoords[(int)*s]+6); glVertex2i(x                 , y + font_charheight);
		}
	}
	glEnd();

	glBindTexture(GL_TEXTURE_2D, 0);
	glDisable(GL_TEXTURE_2D);
	glDisable(GL_BLEND);
}

void drawstringf(int x, int y, const char *s, ...)
{
	static char buffer[16384];
	va_list ap;
	va_start(ap, s);
	vsprintf(buffer, s, ap);
	va_end(ap);
	drawstring(x, y, buffer);
}
