#include "g_local.h"

typedef struct mdl_precache_s mdl_precache_t;

struct mdl_precache_s
	{
	char		file[MAX_QPATH];
	int			frame;
	int			count;
	vec3_t	min;
	vec3_t	max;
	};

/* Open file (use mod directory and then base directory as a
fallback if none is found) */
FILE *fs_fileOpen(char *filename)
	{
	cvar_t	*game_dir;
	FILE		*pFile;
	char		file_name[MAX_QPATH];
	int			res = 1;

	// Try opening file in mod directory
	game_dir = gi.cvar("gamedir", "", 0);
	if (strlen(game_dir->string)) {
		sprintf(file_name, "%s\\%s", game_dir->string, filename);
		res = fopen_s(&pFile, file_name, "rb"); }

	// Try opening file in base directory
	if (res) {
		sprintf(file_name, "main\\%s", filename);
		res = fopen_s(&pFile, file_name, "rb"); }

	// Got a file handle
	if (!res)
		return pFile;

	// File no found
	gi.dprintf("fs_fileOpen: %s no found.\n", filename);
	return NULL;
	}

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *

		New point entity: "misc_model" -- display a model (animated
		or not), and can be displayed or hidden when triggered. Put a
		"clip" brush around the model if you want it to be solid.

		"model"					Model to display
		"targetname"		This entity name
		"frame_first"		First frame
		"frame_last"		Last frame
		"volume"				Alpha (0 to 255 - invisible to solid)
		Spawnflags:			1   start hidden (targetname must be set!)
										2   important! (always display)
										4   non-looped animation
										8   glow
										16  no shadow
										32  dir_light
										64  fullbright
										128 alpha2

* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

#define	MODEL_FLAG_HIDDEN					0x00000001	// Start off
#define	MODEL_FLAG_IMPORTANT			0x00000002	// Important
#define	MODEL_FLAG_NOLOOP					0x00000004	// Non-looped animation
#define	MODEL_FLAG_GLOW						0x00000008	// Glow
#define	MODEL_FLAG_NOSHADOW				0x00000010	// No Shadow
#define	MODEL_FLAG_DIRLIGHT				0x00000020	// Directional Light
#define	MODEL_FLAG_FULLBRIGHT			0x00000040	// Fullbright
#define	MODEL_FLAG_SURFALPHA2			0x00000080	// Alpha2
#define MODEL_FLAG_FLIES          0x00000100  // Emits flies

// Load model file to set proper model BBOX automatically, uses a precache list
// to avoid reloading the same file over and over again. BBOX size based on the
// specified frame, not correction according to the angle!
void setModelBox(edict_t *self, char *material, int frame)
	{
	#define MAX_MODEL_PRECACHE 8
	static mdl_precache_t mdlcache[] = {
		{"", 0, 0, {0,0,0}, {0,0,0} },
		{"", 0, 0, {0,0,0}, {0,0,0} },
		{"", 0, 0, {0,0,0}, {0,0,0} },
		{"", 0, 0, {0,0,0}, {0,0,0} },
		{"", 0, 0, {0,0,0}, {0,0,0} },
		{"", 0, 0, {0,0,0}, {0,0,0} },
		{"", 0, 0, {0,0,0}, {0,0,0} },
		{"", 0, 0, {0,0,0}, {0,0,0} },
		};

	int i, found = -1;
	FILE *pIn;
	int mdlMagic, mdlVersion, frameSize, numVertices, ofsFrames;

	self->solid = SOLID_NOT;
	if (!self->model)
		return;
	if ((!material) || (!strlen(material)))
		return;

	// search existing precache
	for (i = 0; i < MAX_MODEL_PRECACHE; i++)
		{
		if (frame != mdlcache[i].frame)
			continue;
		if (Q_stricmp(self->model, mdlcache[i].file))
			continue;
		VectorCopy(mdlcache[i].min, self->mins);
		VectorCopy(mdlcache[i].max, self->maxs);
		mdlcache[i].count ++;
		found = i;
		break;
		}

	// precache no found, read MD2 file
	if (found == -1)
		{
		pIn = fs_fileOpen(self->model);
		if (!pIn)
			return;
		fread(&mdlMagic, sizeof(mdlMagic), 1, pIn);
		if (mdlMagic != 0x32504449)
			{
			gi.dprintf("setModelBox: Invalid model signature.\n");
			fclose(pIn);
			return;
			}
		fread(&mdlVersion, sizeof(mdlVersion), 1, pIn);
		if (mdlVersion != 8)
			{
			gi.dprintf("setModelBox: Invalid model version.\n");
			fclose(pIn);
			return;
			}
		fseek(pIn, 8, SEEK_CUR);
		fread(&frameSize, sizeof(frameSize), 1, pIn);
		fseek(pIn, 4, SEEK_CUR);
		fread(&numVertices, sizeof(numVertices), 1, pIn);
		fseek(pIn, 28, SEEK_CUR);
		fread(&ofsFrames, sizeof(ofsFrames), 1, pIn);
		fseek(pIn, ofsFrames + frameSize * frame, SEEK_SET);
		fread(self->maxs, sizeof(self->maxs[0]), 3, pIn);
		fread(self->mins, sizeof(self->mins[0]), 3, pIn);
		VectorMA(self->mins, 255, self->maxs, self->maxs);
		fclose(pIn);
		// register precache in least used spot
		found = 0;
		for (i = 0; i < MAX_MODEL_PRECACHE; i++)
			{
			if (mdlcache[i].count <= mdlcache[found].count)
				found = i;
			}
		strcpy(mdlcache[found].file, self->model);
		mdlcache[found].count = 1;
		mdlcache[found].frame = frame;
		VectorCopy(self->mins, mdlcache[found].min);
		VectorCopy(self->maxs, mdlcache[found].max);
		}

	self->solid = SOLID_BBOX;
	if (!Q_stricmp(material, "fabric"))
		self->surfacetype = SURF_FABRIC;
	else if (!Q_stricmp(material, "gravel"))
		self->surfacetype = SURF_GRAVEL;
	else if (!Q_stricmp(material, "metal"))	
		self->surfacetype = SURF_METAL;
	else if (!Q_stricmp(material, "wood"))
		self->surfacetype = SURF_WOOD;
	else
		self->surfacetype = SURF_TILE;
	}

void Anim_Misc_Model (edict_t *self)
	{
	if (self->s.frame < self->frame_last)
		self->s.frame++;
	else
		if (!(self->spawnflags & MODEL_FLAG_NOLOOP))
			self->s.frame = self->frame_first;
	self->nextthink = level.time + FRAMETIME;
	}

void misc_model_setmodel (edict_t *self)
	{
	self->s.modelindex  = gi.modelindex(self->model);
	self->takedamage = DAMAGE_NO;
	//gi.setmodel (self, self->model);
	self->s.frame = self->frame_first;	// reset animation when re-appearing
	if (self->type)
		setModelBox(self, self->type, self->s.frame);
	}

void Use_Misc_Model (edict_t *self, edict_t *other, edict_t *activator)
	{
	self->spawnflags ^= MODEL_FLAG_HIDDEN;
	if (self->spawnflags & MODEL_FLAG_HIDDEN) // hide model
		{
		self->s.frame = 0;
		gi.setmodel (self, "");
		}
	else // show model
		misc_model_setmodel (self);
	}

void SP_misc_model (edict_t *self)
	{
	if (deathmatch->value)	// deathmatch? remove
		{
		G_FreeEdict (self);
		return;
		}

	if (!self->model)	// no model set? remove
		{
		gi.dprintf( "Warning: %s without model at %s\n", self->classname, vtos(self->s.origin) );
		G_FreeEdict (self);
		return;
		}

	if (self->spawnflags & MODEL_FLAG_HIDDEN) // start off?
		{
		if (!self->targetname)	// Start off, but cannot be triggered, remove
			{
			gi.dprintf( "Warning: %s is off, without targetname at %s\n", self->classname, vtos(self->s.origin) );
			G_FreeEdict (self);
			return;
			}
		}
	else	// Start on
		misc_model_setmodel (self);

	if (self->targetname)	// Can be triggered
		self->use = Use_Misc_Model;

	if ((self->frame_last - self->frame_first) > 0)	// we should animate this stuff.
		{
		self->nextthink = level.time + FRAMETIME *2;
		self->think = Anim_Misc_Model;
		}

	if (self->frame_first)	// Set default frame
		self->s.frame = self->frame_first;
	
	if (!(self->spawnflags & MODEL_FLAG_IMPORTANT))
		self->svflags |= SVF_PROP;
	if (self->spawnflags & MODEL_FLAG_GLOW)
		self->s.renderfx = RF_GLOW;
	if (self->spawnflags & MODEL_FLAG_NOSHADOW)
		self->s.renderfx2 |= RF2_NOSHADOW;
	if (self->spawnflags & MODEL_FLAG_DIRLIGHT)
		self->s.renderfx2 |= RF2_DIR_LIGHTS;	// I'm not sure what this stuff is for anymore -- I'm confused (maybe only works with MDX?)
	if (self->spawnflags & MODEL_FLAG_FULLBRIGHT)
		self->s.renderfx |= RF_FULLBRIGHT;		// It's the opposit of RF2_PASSLIGHT
	if (self->spawnflags & MODEL_FLAG_SURFALPHA2)
		self->s.renderfx2 |= RF2_SURF_ALPHA;
	if (self->spawnflags & MODEL_FLAG_FLIES)
		{
		// Flies number increases over time then slowly spread out in 60 to 62 seconds cycles.
		// Destroying a body will make them dissapear at once, which is jarring.
		self->noise_index = gi.soundindex("world/flies.wav");
		self->s.sound = self->noise_index;
		self->s.effects |= EF_FLIES;
		}

	// Uniform light
	if ((self->lightit > 0) && (self->lightit < 32) && (!(self->s.renderfx2 & 31)))
		{
		self->s.renderfx2 &= ~RF2_DIR_LIGHTS;
		self->s.renderfx2 |= RF2_PASSLIGHT;
		self->s.renderfx2 |= self->lightit;
		}

	if (self->volume != 0)						// we use a transparency level, let's set fx flags
		{
		self->s.renderfx2 |= RF2_PASSALPHA;
		self->s.effects = 255 - self->volume;  // reverse values so 0 == visible and 255 == hidden
		}

	if (!self->cast_info.scale)
		self->cast_info.scale = 1.0;

	self->s.scale = self->cast_info.scale - 1.0;

	self->movetype = MOVETYPE_NONE;
	gi.linkentity (self);
	}

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
  
	"target_registerflag"
	Register the specified episode flag when triggered

	"count" - Flag number (1 to 32)
  
* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */
int EP_GetCustomEpisodeFlagValue(char *epflag);

void use_target_epflag_register (edict_t *self, edict_t *other, edict_t *activator)
	{
	EP_Shared_Register_EPFLAG(activator, self->count);
	G_FreeEdict(self);
	}

void SP_target_epflag_register (edict_t *self)
	{
	// Remove in deathmatch
	/*
	if (deathmatch->value)
		{
		G_FreeEdict(self);
		return;
		}*/

	if ((self->count < 1) || (self->count > 32))
		{
		gi.dprintf("%s at %s with an invalid episode flag\n", self->classname, vtos(self->s.origin));
		G_FreeEdict(self);
		return;
		}

	// Usable
	self->count = 1 << (self->count - 1);
	self->use = use_target_epflag_register;
	}

//
void think_target_epflag_execute (edict_t *self)
	{
	edict_t *player;

	// find client, test epflags
	player = &g_edicts[1];
	if (!(player->client->pers.episode_flags & self->count))
		return;

	// use targets and remove self
	G_UseTargets (self, self);
	G_FreeEdict (self);
	}

void SP_target_epflag_execute (edict_t *self)
	{
	// Remove in deathmatch
	/*
	if (deathmatch->value)
		{
		G_FreeEdict(self);
		return;
		}*/

	// Check flag
	if ((self->count < 1) || (self->count > 32))
		{
		gi.dprintf("%s at %s with an invalid episode flag\n", self->classname, vtos(self->s.origin));
		G_FreeEdict(self);
		return;
		}

	// Usable
	self->delay = 1;
	self->count = 1 << (self->count - 1);
	self->svflags = SVF_NOCLIENT;
	self->think = think_target_epflag_execute;
	self->nextthink = level.time + self->delay;
	}




/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
	
	"target_showhelp"
	Displays the notepad icon when triggered, similarly to how it is displayed in
	"sr1", "pv_h", "sy_h", "steel1," ty1" and "rc1".

* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */
void Show_Help (void);

void use_target_showhelp (edict_t *self, edict_t *other, edict_t *activator)
	{
	if (!activator->client)
		return;
	Show_Help();
	G_FreeEdict(self);
	}

void SP_target_showhelp (edict_t *self)
	{
	if (deathmatch->value)
		{
		G_FreeEdict(self);
		return;
		}

	self->use = use_target_showhelp;
	}

#define MMC_FLIES				0x00000001
#define MMC_SPAWN				0x00000002
#define MMC_FASTREMOVE  0x00000004
#define MMC_REMOVESELF	0x00000008
#define MMC_NOSHADOW		0x00000010
#define MMC_DIRLIGHTS		0x00000020
#define MMC_FULLBRIGHT	0x00000040

/*
art_skins: head, torso and legs (three digits each)
type: thug, punk, runt, shorty, bitch, whore
head: 1 - bald, 2 - pony/weld, 3 - headless (any other value is default head for that model)
message: weapon (pipe for thug etc.)
spawnflags: 
1 - flies
2 - trigger spawn
8 - delete on last frame
16 - no shadow
32 - uses junior lights
64 - full bright
*/
void misc_model_cast_idle(edict_t *self)
	{
	if (self->s.frame < self->frame_last)
		self->s.frame++;
	else
		{
		// we're expected to trigger something at the end of the count
		if (self->count > 0)
			self->count--;

		// no more loops
		if (!self->count)
			{
			if (self->target || self->killtarget) // fire targets
				{
				G_UseTargets (self, self);
				self->think = NULL;
				}

			if (self->spawnflags & MMC_REMOVESELF) // remove self
				{
				// test for fast removal
				if (self->target)
					{
					edict_t *ent = NULL;

					while ((ent = G_Find(ent, FOFS(targetname), self->target)) != NULL)
						{
						if (!Q_stricmp(ent->classname, self->classname))
							{
							G_FreeEdict(self);
							return;
							}
						}
					}
				// normal remove
				self->nextthink = level.time + 0.1;
				self->think = G_FreeEdict;
				return;
				}
			}
		else // count not null, return to initial frame
			self->s.frame = self->frame_first;
		}
	self->nextthink = level.time + FRAMETIME;
	}

void misc_model_cast_use (edict_t *self, edict_t *other, edict_t *activator)
	{
	// no last frame specified, assume static
	if (!self->frame_last)
		self->frame_last = self->frame_first;

	// if there's more than one frame of animation, animate it
	if ((self->frame_last - self->frame_first) > 0)
		{
		self->nextthink = level.time + 0.1;
		self->think = misc_model_cast_idle;
		}
	self->s.frame = self->frame_first;

	// spawnflags
	if (self->spawnflags & MMC_NOSHADOW)
		self->s.renderfx2 |= RF2_NOSHADOW;
	if (self->spawnflags & MMC_DIRLIGHTS)
		self->s.renderfx2 |= RF2_DIR_LIGHTS;
	if (self->spawnflags & MMC_FULLBRIGHT)
		self->s.renderfx |= RF_FULLBRIGHT;
	if (self->spawnflags & MMC_FLIES)
		{
		self->noise_index = gi.soundindex("world/flies.wav");
		self->s.sound = self->noise_index;
		self->s.effects |= EF_FLIES;
		}

	// ready entity
	self->svflags &= ~SVF_NOCLIENT;
	gi.linkentity (self);
	}

void SP_misc_model_cast(edict_t *self)
	{
	int i;
	char artskin[11];
	char filemask[256], filefull[256];
	char *partSkin;
	int partIndex;
	int	skin;

	// not in deathmatch
	if (deathmatch->value)
		{
		G_FreeEdict (self);
		return;
		}

	if (self->model)
		{
		gi.dprintf("misc_model_cast with an explicit model field at %s!\n", vtos(self->s.origin));
		G_FreeEdict (self);
		return;
		}

	if (!self->type)
		{
		gi.dprintf("misc_model_cast without model type at %s!\n", vtos(self->s.origin));
		G_FreeEdict (self);
		return;
		}

	// make dummy non-solid
	self->solid = SOLID_NOT;
	self->takedamage = DAMAGE_NO;

	// get skins (quick & dirty)
	if (self->art_skins)
		{
		strcpy(artskin, self->art_skins);
		artskin[3] = artskin[7] = '\0';
		}
	else
		artskin[0] = artskin[4] = artskin[8] = '\0';

	// get file mask
	sprintf(filemask, "models/actors/%s/", self->type);
	strcat(filemask, "%s.mdx");

	// setup body
	memset(&(self->s.model_parts[0]), 0, sizeof(model_part_t) * MAX_MODEL_PARTS);
	for (partIndex = PART_HEAD; partIndex <= PART_GUN; partIndex++)
		{
		switch (partIndex)
			{
			case PART_HEAD:
				sprintf(filefull, filemask, "head");
				switch (self->head)
					{
					case 1:
						if (!strcmp(self->type, "thug") || !strcmp(self->type, "punk") || !strcmp(self->type, "whore") || !strcmp(self->type, "bitch") || !strcmp(self->type, "thug_sit"))
							sprintf(filefull, filemask, "bald_head");
						break;
					case 2:
						if (!strcmp(self->type, "thug") || !strcmp(self->type, "punk") || !strcmp(self->type, "thug_sit"))
							sprintf(filefull, filemask, "weld_head");
						else if (!strcmp(self->type, "whore") || !strcmp(self->type, "bitch"))
							sprintf(filefull, filemask, "pony_head");
						break;
					case 3:
						continue;
					}
				partSkin = &artskin[0];
				break;
			case PART_LEGS:
				sprintf(filefull, filemask, "legs");
				partSkin = &artskin[8];
				break;
			case PART_BODY:
				sprintf(filefull, filemask, "body");
				partSkin = &artskin[4];
				break;
			case PART_GUN:
				if (!self->message)
					continue;
				sprintf(filefull, filemask, self->message);
				skin = 0;
				break;
			}
		self->s.num_parts++;
		self->s.model_parts[partIndex].modelindex = gi.modelindex(filefull);
		if (partIndex != PART_GUN)
			{
			if (partSkin > 0)
				skin = gi.skinindex(self->s.model_parts[partIndex].modelindex, partSkin);
			else
				skin = self->s.skinnum;
			}
		for (i = 0; i < MAX_MODELPART_OBJECTS; i++)
			self->s.model_parts[partIndex].baseskin = self->s.model_parts[partIndex].skinnum[i] = skin;
		gi.GetObjectBounds(filefull, &self->s.model_parts[partIndex]);
		}

	// Infinite loop if no count specified
	if (!self->count)
		self->count = -1;

	// We're supposed to fire targets at the end of the loop count, make sure we
	// play at least 1, also figure out if we should be in fast remove mode or not.
	self->spawnflags &= ~MMC_FASTREMOVE;
	if (self->target || self->killtarget)
		{
		if (self->count < 1)
			self->count = 1;
		}

	// trigger spawn
	if (self->spawnflags & MMC_SPAWN)
		{
		self->solid = SOLID_NOT;
		self->movetype = MOVETYPE_NONE;
		self->svflags |= SVF_NOCLIENT;
		self->nextthink = 0;
		self->use = misc_model_cast_use;
		return;
		}

	// initialize right away
	misc_model_cast_use(self, NULL, NULL);
	}

// this code spawns a temporary entity that plays a sound, lives for a while and then removes itself.
// it's useful for breaking sound as the original entity won't play a sound if it's being removed.
void generic_sound_entity_play(edict_t *self)
	{
	gi.sound (self, CHAN_VOICE, self->count, 1, ATTN_NORM, 0);
	self->think = G_FreeEdict;
	self->nextthink = level.time + self->health;
	}

void generic_sound_entity(vec3_t org, char *noise, int duration)
	{
	edict_t *temp;

	temp = G_Spawn();
	if (!temp)
		return;

	temp->think = generic_sound_entity_play;
	temp->nextthink = level.time + 0.1;
	temp->count = gi.soundindex(noise);
	temp->health = duration;
	if (!temp->health)
		temp->health = 2;
	VectorCopy (org, temp->s.origin);
	gi.linkentity(temp);
	}

void trashcanA_check_sound (edict_t *ent);

void stool_push_die (edict_t *self, edict_t *inflictor, edict_t *attacker, int damage, vec3_t point, int mdx_part, int mdx_subobject)
	{
	vec3_t	org, mid;
	float		vel;
	int			cnt = 5;

	self->takedamage = DAMAGE_NO;
	self->activator = attacker;

	// spawn temporary sound entity
	generic_sound_entity (self->s.origin, "world/boardbreak.wav", 2);

	// spawn debris
	VectorMA (self->absmin, 0.5, self->size, mid);
	vel = 1.5 * (float)self->dmg / 200.0;
	while (cnt--)
		{
		org[0] = mid[0] + crandom() * self->size[0];
		org[1] = mid[1] + crandom() * self->size[1];
		org[2] = mid[2] + crandom() * self->size[2];
		ThrowDebrisScale (self, "models/props/wood/wood1.md2", vel, org, .5);
		}

	// remove entity
	G_FreeEdict (self);
	}

void stool_push_touch (edict_t *self, edict_t *other, cplane_t *plane, csurface_t *surf)
	{
	float	ratio;
	vec3_t	v;

	if ((!other->groundentity) || (other->groundentity == self))
		return;

	self->pullingflag = 0;

	// If activate key is pressed
	if ((plane) && (plane->type == 123))
		{
		if ((other->s.origin[0] != other->s.old_origin[0]) || (other->s.origin[1] != other->s.old_origin[1]))
			{
			self->pullingflag = 1;
			
			ratio = (float)other->mass / (float)self->mass;
			VectorSubtract (self->s.origin, other->s.origin, v);
			M_walkmove (self, 180+vectoyaw(v), 20 * ratio * FRAMETIME);
		
			if (!self->s.sound)
				self->s.sound = gi.soundindex ("world/crate1.wav");
			VectorCopy(self->s.origin, self->save_avel);
			self->think = trashcanA_check_sound;
			self->nextthink = level.time + (FRAMETIME * 1.1);
			}
		}
	else
		{	
		ratio = (float)other->mass / (float)self->mass;
		VectorSubtract (self->s.origin, other->s.origin, v);
		M_walkmove (self, vectoyaw(v), 20 * ratio * FRAMETIME);

		if (((self->s.origin[0] != self->s.old_origin[0]) || (self->s.origin[1] != self->s.old_origin[1])))
			{
			if (!self->s.sound)
				self->s.sound = gi.soundindex ("world/crate1.wav");	
			VectorCopy(self->s.origin, self->save_avel);
			self->think = trashcanA_check_sound;
			self->nextthink = level.time + (FRAMETIME * 1.1);
			}
		}

	if (self->health <= 0)
		stool_push_die(self, NULL, NULL, 0, vec3_origin, 0, 0);
	}

void SP_props4_stool_push (edict_t *self)
	{
	if (deathmatch->value)
		{
		G_FreeEdict (self);
		return;
		}
	
	self->solid = SOLID_BBOX;
	self->movetype = MOVETYPE_STEP;
	self->svflags |= SVF_PROP;

	self->pullable = 1;
	self->nokickbackflag = 1;

	self->model = "models/props/stool/tris.md2";
	self->s.modelindex = gi.modelindex (self->model);
	VectorSet (self->mins, -16, -16, -16);
	VectorSet (self->maxs, 16, 16, 16);		

	if (!self->mass)
		self->mass = 50;	

	if (!self->health)
		self->health = 25;

	self->die = stool_push_die;
	self->takedamage = DAMAGE_YES;

	self->cast_info.aiflags = AI_NOSTEP;
	self->touch = stool_push_touch;
	
	self->think = M_droptofloor;
	self->nextthink = level.time + 2 * FRAMETIME;

	self->surfacetype = SURF_WOOD;
	gi.linkentity (self);
	}


void trigger_thunder_initial (edict_t *self);

#if 1
void trigger_thunder_flash (edict_t *self)
	{
	if (self->acc) // initial sound
		{
		self->acc = 0;
		gi.positioned_sound(self->s.origin, self, CHAN_VOICE, gi.soundindex("world/event/thunder.wav"), 1.00, 0, 0);
		self->nextthink = level.time + 2 + random(); // wait between 2 and 3 seconds for first flash.
		}
	else // flash
		{
		float rnd;
		self->count --;
		G_UseTargets (self, self->activator);

		// more flashes?
		if (self->count)
			{
			if (self->count & 1)
				{
				rnd = random() * (self->count / 7); // light duration
				if (self->style)
					{
					if (rnd > .45)
						rnd = .45;
					}
				else
					{
					if (rnd > 0.90)
						{
						rnd = 0.90;
						self->style = 1;
						}
					}
				}
			else
				{
				rnd = random() * (self->count / 4); // darkness duration
				if (self->style)
					{
					if (rnd > .60)
						rnd = .60;
					}
				else
					{
					if (rnd > 1.20)
						{
						rnd = 1.20;
						self->style = 1;
						}
					}
				}
			if (!rnd)
				rnd = 0.01;
			self->nextthink = level.time + rnd;
			}
		else // no more flashes, return to initial status
			{
			self->nextthink = level.time + 0.10;
			self->think = trigger_thunder_initial;
			}
		}
	}

void trigger_thunder_initial (edict_t *self)
	{
	// switch to light trigger
	self->nextthink = level.time + self->wait + random() * self->random;
	self->think = trigger_thunder_flash;
	// number of triggers (must be even)
	self->count = (random() * 3 + 3) * 2;
	self->count += (self->count & 1);
	self->acc = 1; // play sound
	self->style = 0; // reset "long strike"
	}
#else
void trigger_thunder_flash (edict_t *self)
	{
	static float strike[] = { 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0 };

	if (self->acc) // initial strike
		{
		int i, c = 0;
		float d = random() * 3 + 1; // duration

		for (i = 0; i < self->count; i++)
			{
			strike[i] = (int)(random() * 32) + 4; // random sequence of light and darkness
			if (i & 1)
				strike[i] = (int)(strike[i] * 1.10);
			c += strike[i];
			}
		for (i = 0; i < self->count; i++)
			strike[i] = (strike[i] / c) * d; // rescale whole sequence to fit within duration

		self->acc = 0;
		gi.positioned_sound(self->s.origin, self, CHAN_VOICE, gi.soundindex("world/event/thunder.wav"), 1.00, 0, 0);
		self->nextthink = level.time + 1.5 + random(); // wait between 1.5 and 2.5 seconds for first flash.
		}
	else // flash
		{
		G_UseTargets (self, self->activator);
		self->count --;
		if (self->count)
			self->nextthink = level.time + strike[self->count];
		else // no more flashes, return to initial status
			{
			self->nextthink = level.time + 0.10;
			self->think = trigger_thunder_initial;
			}
		}
	}

void trigger_thunder_initial (edict_t *self)
	{
	// switch to light trigger
	self->nextthink = level.time + self->wait + random() * self->random;
	self->think = trigger_thunder_flash;
	// number of triggers (must be even)
	self->count = (int)(random() * 3 + 3) * 2;
	// start fresh
	self->acc = 1;
	}
#endif
// This could easily be done with func_timer and trigger_relay, but it's also a pain the manage. This
// this is much easier to handle in maps: one trigger and a bunch of lights and we're good. No need
// to include a target_speaker either (if I can clean up some more thunder sound, I could also add
// variations)
// "wait" is the base delay between two lightning strikes
// "random" is the extra random delay that can happen, added on top of "wait"
// Sound will play between 1.5 and 3 seconds BEFORE the lightning strike.
void SP_trigger_thunder (edict_t *self)
	{
	if (!self->wait)
		self->wait = 1.0;

	self->think = trigger_thunder_initial;
	self->nextthink = level.time + 1.0 + st.pausetime;
	
	self->activator = self;
	self->svflags = SVF_NOCLIENT;
	}


// This works like trigger_counter as it fires its targets once the counter
// reaches the specified value, however it will reset so it can fire its
// targets again. It's a very specific thing...
void multi_trigger (edict_t *ent);

void trigger_counter_cycle_use(edict_t *self, edict_t *other, edict_t *activator)
	{
	self->health++;

	if (self->health == self->count)
		{
		self->activator = activator;
		multi_trigger(self);
		}

	if (self->health == self->dmg)
		self->health = self->acc; // self->acc should be 0, unless we're meant to reset to some other value
	}

void SP_trigger_counter_cycle (edict_t *self)
	{
	if (!self->count)
		self->count = 2; // fires its targets when COUNT activations are reached
	if (self->dmg < self->count)
		self->dmg = self->count; // resets counter when DMG activations are reached
	self->health = 0; // initial activations count
	self->wait = FRAMETIME; // we need a delay or multi_trigger will disable the entity;
	self->use = trigger_counter_cycle_use;
	}

#if 0
/*********************************************************************
		TRIGGER_SWITCHAREA - 20th may 2011

		New brush entity: "trigger_switcharea" -- when the player
		enter or exit a marked area, an entity can be triggered and/or
		an episode flag can be toggled.
	
		SPAWNFLAGS:
		1	Start IN (player starts inside the area)
		2	Do not trigger when going in
		4	Do not trigger when going out
		8	Master

		"target"		Entity to trigger
		"targetname"	The area's name
*********************************************************************/

void G_LinkAreaMarkers (edict_t *self)
	{
	edict_t		*parentent = NULL;

	if (!self->targetname)
		return;

	while ((parentent = G_Find (parentent, FOFS(targetname), self->targetname)))
		{
		if ((parentent != self) && (parentent->count != self->count))
			parentent->count = self->count;
		}
	}

void trigger_switcharea_touch (edict_t *self, edict_t *other, cplane_t *plane, csurface_t *surf)
	{
	vec3_t		vec;
	int				AllowAct = 0;
	edict_t		*player = &g_edicts[1];

	if (!other->client)
		return;

	VectorSubtract(other->s.origin, self->pos1, vec);	// Are we going into or out of the area?
	vec[2] = 0;
	VectorNormalize(vec);

	if (DotProduct(vec, self->movedir) > 0)	// Going in
		{
		if (!self->count)
			{
			self->count = 1;
			G_LinkAreaMarkers (self);	// Link all other area markers
			if (!(self->spawnflags & 2))
				AllowAct = 1;
			else
				gi.dprintf("Going in with NO_IN flag\n");
			}
		}
	else	// Going out
		{
		if (self->count)
			{
			self->count = 0;
			G_LinkAreaMarkers (self);	// Link all other area markers
			if (!(self->spawnflags & 4))
				AllowAct = 1;
			else
				gi.dprintf("Going out with NO_OUT flag\n");
			}
		}

	if ((AllowAct == 1) && (self->target))
		G_UseTargets (self, other);
	}

void SP_trigger_switcharea (edict_t *ent)
	{
	if (deathmatch->value)
		{
		G_FreeEdict(ent);
		return;
		}

	ent->solid = SOLID_TRIGGER;
	ent->touch = trigger_switcharea_touch;
	ent->svflags |= SVF_NOCLIENT;

	if (!ent->target)
		gi.dprintf("%s without target at %s!\n", ent->classname, vtos(ent->s.origin));
	if ((ent->spawnflags & 2) && (ent->spawnflags & 4))
		gi.dprintf("useless %s at %s!\n", ent->classname, vtos(ent->s.origin));

	ent->count = ent->spawnflags & 1; // Set default position to "in area" (1) or "out of area" (0)
	if (ent->spawnflags & 8)
		{
		if (!(ent->targetname))
			gi.dprintf("%s at %s set as MASTER, but has no targetname!\n", ent->classname, vtos(ent->s.origin));
		else
			{
			ent->think = G_LinkAreaMarkers;				// Make sure all other area markers are set like this one
			ent->nextthink = level.time + 0.1;		// Wait a bit beofre linking all markers
			}
		}

	gi.setmodel (ent, ent->model);
	gi.linkentity (ent);

	VectorAdd(ent->absmin, ent->absmax, ent->pos1);
	VectorScale(ent->pos1, 0.5, ent->pos1);
	AngleVectors(ent->s.angles, ent->movedir, NULL, NULL);
	}
#endif

void think_motorcycle_drives(edict_t *self)
	{
	// should use path_corner, but I'm tight on schedule, it'll have to do (just update origin)
	VectorMA(self->s.origin, self->speed, self->movedir, self->s.origin); // s.origin + speed * movedir --> s.origin
	self->nextthink = level.time + 0.1;
	}

void use_motorcycle_drive (edict_t *self, edict_t *other, edict_t *activator)
	{
	AngleVectors(self->s.angles, self->movedir, NULL, NULL);
	self->think = think_motorcycle_drives;
	self->nextthink = level.time + 0.1;
	}

void SP_props4_motorcycle_drive (edict_t *self)
	{
	int		i, p;
	char	filefull[256], *list[4] = {"head", "moto", "body", "legs"};

	if (deathmatch->value)
		{
		G_FreeEdict (self);
		return;
		}

	self->solid = SOLID_NOT;
	self->movetype = MOVETYPE_NOCLIP;
	self->svflags |= SVF_PROP;

	self->s.skinnum = self->skin;
 	memset(&(self->s.model_parts[0]), 0, sizeof(model_part_t) * MAX_MODEL_PARTS);
	
	for (p = 0; p < 4; p++)
		{
		sprintf(filefull, "models/props/runaway/%s.mdx", list[p]);

		self->s.num_parts++;
		self->s.model_parts[p].modelindex = gi.modelindex(filefull);
		for (i=0; i<MAX_MODELPART_OBJECTS; i++)
			self->s.model_parts[p].skinnum[i] = self->s.skinnum;
		gi.GetObjectBounds(filefull, &self->s.model_parts[p]);	
		}

	self->s.renderfx2 |= RF2_DIR_LIGHTS;
	self->s.renderfx  |= RF_REFL_MAP;

	self->use = use_motorcycle_drive;

	gi.linkentity (self);
	}

void touch_misc_ladies_bathroom (edict_t *self, edict_t *other, cplane_t *plane, csurface_t *surf)
	{
	game.serverflags |= SFL_MOMO_LADIES;
	self->think = G_FreeEdict;
	self->nextthink = level.time + 0.1;
	}
	
void SP_misc_ladies_bathroom (edict_t* self)
	{
	self->movetype = MOVETYPE_NONE;
	self->svflags |= SVF_NOCLIENT;
	self->solid = SOLID_TRIGGER;

	self->touch = touch_misc_ladies_bathroom;

	gi.setmodel (self, self->model);
	gi.linkentity (self);
	}