//CTF_flag.c - Routines dealing with team flags

#include "g_local.h"

char* Client_Crashed_String = "AUTO FLAG RESPAWN\n";

char* CTF_Flag_Class_Name = "ctf_flag";

typedef struct _Flag_Initial_Spawn_Def
{
	vec3_t Position;
	vec3_t Angle;
	qboolean Exists;		//FALSE if flag not found
} Flag_Initial_Spawn_Def;

Flag_Initial_Spawn_Def Flag_Initial_Spawn[2];	//0 = team 1, 1 = team 2

//Forwards
void CTF_Flag_Respawn(edict_t *CarryingEnt, int Team, Respawn_Type CTF_Flag_Respawn);
void ctf_flag_touch (edict_t *self, edict_t *other, cplane_t *plane, csurface_t *surf);
void ctf_flag_think (edict_t *self);

void CTF_Flag_Initialise(void)
{//Called at map load
	Flag_Initial_Spawn[0].Exists = FALSE;
	Flag_Initial_Spawn[1].Exists = FALSE;
}

void CTF_Match_Start_Reset_Flags(void)
{
	CTF_Flag_Respawn(NULL, 1, CTF_FLAG_RESPAWN_MATCH_START);
	CTF_Flag_Respawn(NULL, 2, CTF_FLAG_RESPAWN_MATCH_START);
}

char* GetTeamName(edict_t *ent, TeamNameType WhichTeam)
{
	if(!ent)
		return NULL;
	if(!ent->client)
		return NULL;
	switch(WhichTeam)
	{
		case TEAMNAME_OWNTEAM:
		{
			if(ent->client->pers.team == 1)
				return team_names[1];
			else
				return team_names[2];
			break;
		}
		case TEAMNAME_OTHERTEAM:
		{
			if(ent->client->pers.team == 2)
				return team_names[1];
			else
				return team_names[2];
			break;
		}
	}
	return NULL;
}

qboolean CTF_Flag_At_Base(edict_t *ent)
{
	edict_t *find = NULL;
	Flag_Initial_Spawn_Def* Flag_Initial;

//Get the initial parameters of the incoming flag style
	Flag_Initial = &Flag_Initial_Spawn[ent->style - 1];

	while ((find = G_Find(find, FOFS(classname), CTF_Flag_Class_Name)) != NULL)
	{
		if(!find)
			return false;

		if(find->style != ent->style)
			continue;

		if (VectorCompare(Flag_Initial->Position, find->s.origin))
			return true;
	}
	return false;
}

edict_t* CTF_Flag_Find(int Team)
{
	edict_t *find = NULL;

	while ((find = G_Find(find, FOFS(classname), CTF_Flag_Class_Name)) != NULL)
	{
		if(!find)
			return NULL;

		if(find->style != Team)
			continue;
		return find;
	}
	return NULL;
}

//Check in case a player carrying the flag has crashed so we can return the flag
//This will not check for a lagged player
//Dont check every frame though
//****ADD IN CALL TO THIS IN MAIN.C****
void CTF_Crashed_Player_Check(void)
{
	int Count;
	qboolean FoundTeam1;
	qboolean FoundTeam2;
	edict_t *Flag;
	edict_t *ent;

	CTF_Game.CrashedPlayerCheckTime--;
	if(CTF_Game.CrashedPlayerCheckTime > 0)
		return;
	CTF_Game.CrashedPlayerCheckTime = CTF_CRASHED_PLAYER_CHECK_DELAY;

	FoundTeam1 = false;
	FoundTeam2 = false;
	for_each_player(ent, Count)
	{
		if (ent->client->resp.PlayerData.Flag_State != CTF_DATA_FLAG_STATE_NULL)
			if(ent->client->pers.team == 1)
				FoundTeam2 = true;	//Team 1 player carrying team 2 flag
			else
				FoundTeam1 = true;	//Team 2 player carrying team 1 flag
	}
	if(FoundTeam1 == false)
	{
//no-one has team 1 flag
		Flag = CTF_Flag_Find(1);
		if(!Flag)
//No team flag was found
		{
			CTF_Flag_Respawn(NULL, 1, CTF_FLAG_RESPAWN_PLAYER_CRASHED);
		}
	}
	if(FoundTeam2 == false)
	{
//no-one has team 2 flag
		Flag = CTF_Flag_Find(2);
		if(!Flag)
//No team flag was found
		{
			CTF_Flag_Respawn(NULL, 2, CTF_FLAG_RESPAWN_PLAYER_CRASHED);
		}
	}
}

void CTF_Check_Award_Defend_Bonus(edict_t *target, edict_t *attacker)
{
	edict_t *find = NULL;
//Call this from death routine in P_Client.c

	find = CTF_Flag_Find(attacker->client->pers.team);

//Get the attackers flag ent - will not be found if an enemy is carrying it
	if (!find)
		return;

//Should probably check if flag is at base as well
	if (CTF_Flag_At_Base(find) == false)
		return;
//Was it visible to the enemy?
	if (!visible(find, target))
		return;
//Yes - is it within a certain distance (for open maps)
	if (VectorDistance(find->s.origin, target->s.origin) > 512.0f)
		return;
//Its within the allowed distance - award a bonus
	gi.bprintf(PRINT_HIGH, "%s defends the %s flag.\n", attacker->client->pers.netname, GetTeamName(attacker, TEAMNAME_OWNTEAM));
	attacker->client->resp.score += CTF_POINTS_FLAG_PROTECT_BONUS;
}

qboolean CTF_Check_Award_Kill_Bonus(edict_t *target, edict_t *attacker)
{
	if (!attacker)
		return false;
	if (target->client->resp.PlayerData.Flag_State == CTF_DATA_FLAG_STATE_NULL)
		return false;
	gi.bprintf(PRINT_HIGH, "%s was carrying the flag. %s gets a kill bonus.\n", target->client->pers.netname, attacker->client->pers.netname);
	attacker->client->resp.score += CTF_POINTS_CARRIER_KILL_BONUS;
	return true;
}

void CTF_Flag_Set_Base_Spawn_Parameters(edict_t* ent, int Team)
{//Sets base flag parameters. Called from multiple locations

	ent->model = CTF_Flag_Model_Array[Team - 1];
	ent->style = Team;
	ent->classname = CTF_Flag_Class_Name;

	ent->s.modelindex = CTF_Game.Flag_Model_Index[Team - 1];

	VectorSet(ent->mins, -13, -13, -26);
	VectorSet(ent->maxs, 37, 37, 74);

	ent->solid = SOLID_TRIGGER;
	ent->s.renderfx = RF_MINLIGHT;
	ent->touch = ctf_flag_touch;
	ent->nextthink = level.time + FLAG_THINK_TIME;
	ent->think = ctf_flag_think;
//New
//Set just in case of lerp from old location (origin)
	VectorCopy(ent->s.origin, ent->s.old_origin);
//Required when cast members are active with holstered weapons. Engine must set render values only when something is defined.
	ent->s.num_parts = 1;
	ent->s.model_parts[0].modelindex = ent->s.modelindex;
	ent->s.model_parts[0].baseskin = 0;
	ent->s.model_parts[0].skinnum[0] = 0;

	gi.linkentity(ent);
}

void CTF_Flag_Respawn(edict_t *CarryingEnt, int Team, Respawn_Type Respawn)
{//Incoming Team is the team of the ent carrying the flag, or the required team flag
	edict_t *New_Flag = NULL;
	edict_t	*Existing_Flag = NULL;
	int Enemy_Team;
	Flag_Initial_Spawn_Def* Flag_Initial;

//Sanity
	if ((Respawn == CTF_FLAG_RESPAWN_NULL) || (Respawn >= CTF_FLAG_RESPAWN_OUT_OF_RANGE))
		return;

//Spawn a new flag
	New_Flag = G_Spawn();
	switch (Respawn)
	{
		case CTF_FLAG_RESPAWN_RESCUED:
		{
//Flag is somewhere in the map
//Delete the team flag if it exists
			Existing_Flag = CTF_Flag_Find(Team);
			if (Existing_Flag)
			{
//				gi.unlinkentity(Existing_Flag);
				G_FreeEdict(Existing_Flag);
			}
			New_Flag->CTF_Flag.Dropped = false;
			New_Flag->movetype = MOVETYPE_NONE;

			Flag_Initial = &Flag_Initial_Spawn[Team - 1];
			VectorCopy(Flag_Initial->Position, New_Flag->s.origin);
			VectorCopy(Flag_Initial->Angle, New_Flag->s.angles);

//			VectorCopy(FlagInitialSpawnPoint[OwnTeam - 1], NewFlag->s.origin);
//			VectorCopy(FlagInitialSpawnAngle[OwnTeam - 1], NewFlag->s.angles);
			gi.sound(New_Flag, CHAN_NO_PHS_ADD + CHAN_VOICE, CTF_Game.Flag_Return_Sound, 1.0f, ATTN_NONE, 0);
			break;
		}
		case CTF_FLAG_RESPAWN_CAPTURED:
		{
//No flag entity exists as a player was carrying it
//Want to respawn the opposite team flag
			if (Team == 1)
				Team = 2;
			else
				Team = 1;
			New_Flag->CTF_Flag.Dropped = false;
			New_Flag->movetype = MOVETYPE_NONE;//MOVETYPE_TOSS;

			Flag_Initial = &Flag_Initial_Spawn[Team - 1];
			VectorCopy(Flag_Initial->Position, New_Flag->s.origin);
			VectorCopy(Flag_Initial->Angle, New_Flag->s.angles);

//			VectorCopy(FlagInitialSpawnPoint[EnemyTeam - 1], NewFlag->s.origin);
//			VectorCopy(FlagInitialSpawnAngle[EnemyTeam - 1], NewFlag->s.angles);
			gi.sound(New_Flag, CHAN_NO_PHS_ADD + CHAN_VOICE, CTF_Game.Flag_Capture_Sound, 0.70f, ATTN_NONE, 0);
			break;
		}
		case CTF_FLAG_RESPAWN_DROPPED:
		{
//No flag entity exists as a player was carrying it
//Requires the ent that is currently carrying it to be passed
//Want to respawn the opposite team flag
			if (Team == 1)
				Team = 2;
			else
				Team = 1;
			New_Flag->CTF_Flag.Dropped = true;
			New_Flag->CTF_Flag.ReturnTimer = FLAG_RETURN_DELAY;
			New_Flag->movetype = MOVETYPE_TOSS;

			VectorCopy(CarryingEnt->s.origin, New_Flag->s.origin);
			VectorCopy(CarryingEnt->s.angles, New_Flag->s.angles);
/*			New_Flag->velocity[0] = crandom() * 100;
			New_Flag->velocity[1] = crandom() * 100;
			if (New_Flag->velocity[0] < 0)
				New_Flag->velocity[0] -= 20;
			else
				New_Flag->velocity[0] += 20;
			if (New_Flag->velocity[1] < 0)
				New_Flag->velocity[1] -= 20;
			else
				New_Flag->velocity[1] += 20;
*/
//Only vertical velocity required now.
			New_Flag->velocity[0] = 0.0f;
			New_Flag->velocity[1] = 0.0f;
			New_Flag->velocity[2] = 300.0f;
//Spawn at players body height
			New_Flag->s.origin[2] += 30;
			break;
		}
		case CTF_FLAG_RESPAWN_MATCH_START:
		{
//Flag is somewhere in the map
//Delete the team flag if it exists
			Existing_Flag = CTF_Flag_Find(Team);
			if (Existing_Flag)
			{
				gi.unlinkentity(Existing_Flag);
				G_FreeEdict(Existing_Flag);
			}
			Flag_Initial = &Flag_Initial_Spawn[Team - 1];
//Check if a flag for this team ever existed
			if (Flag_Initial->Exists == false)
			{//Map does not have a spawn for this flag so delete it
				G_FreeEdict(New_Flag);
				return;
			}
			New_Flag->CTF_Flag.Dropped = false;
			New_Flag->movetype = MOVETYPE_NONE;

			VectorCopy(Flag_Initial->Position, New_Flag->s.origin);
			VectorCopy(Flag_Initial->Angle, New_Flag->s.angles);
			break;
		}
		case CTF_FLAG_RESPAWN_PLAYER_CRASHED:
		{
			Flag_Initial = &Flag_Initial_Spawn[Team - 1];
//Check if a flag for this team ever existed
			if (Flag_Initial->Exists == false)
			{//Map does not have a spawn for this flag so delete it
				G_FreeEdict(New_Flag);
				return;
			}
//No flag entity exists as a player was carrying it. The player entity no longer exists so it is determined elsewhere which flag to spawn
			New_Flag->CTF_Flag.Dropped = false;
			New_Flag->movetype = MOVETYPE_NONE;

			VectorCopy(Flag_Initial->Position, New_Flag->s.origin);
			VectorCopy(Flag_Initial->Angle, New_Flag->s.angles);

//			VectorCopy(FlagInitialSpawnPoint[OwnTeam - 1], NewFlag->s.origin);
//			VectorCopy(FlagInitialSpawnAngle[OwnTeam - 1], NewFlag->s.angles);
			gi.sound(New_Flag, CHAN_NO_PHS_ADD + CHAN_VOICE, CTF_Game.Flag_Return_Sound, 1.0f, ATTN_NONE, 0);
			gi.bprintf(PRINT_HIGH, Client_Crashed_String);
//			gi.dprintf("Flag RESPAWN: %d\n", Team);
			break;
		}
	}
//Set up the common parameters and link
	CTF_Flag_Set_Base_Spawn_Parameters(New_Flag, Team);
}

void CTF_Flag_Player_Throw(edict_t *ent)
{//Player died or quits while carrying the enemy flag
	if (ent->client->resp.PlayerData.Flag_State != CTF_DATA_FLAG_STATE_NULL)
	{
		gi.bprintf(PRINT_HIGH, "%s dropped the %s flag.\n", ent->client->pers.netname, GetTeamName(ent, TEAMNAME_OTHERTEAM));
		CTF_Flag_Respawn(ent, ent->client->pers.team, CTF_FLAG_RESPAWN_DROPPED);
		ent->client->resp.PlayerData.Flag_State = CTF_DATA_FLAG_STATE_NULL;
	}
}

void AwardTeamBonus(int Team)
{
	edict_t *ent;
	int Count;

	ent = g_edicts + 1;
	for (Count = 0; Count<game.maxclients; Count++, ent++)
	{
		if(!ent)
			continue;
		if(!ent->inuse)
			continue;
		if(ent->client->pers.team == Team)
			ent->client->resp.score += CTF_POINTS_TEAM_CAPTURE_BONUS;
	}
}

void ctf_flag_touch (edict_t *self, edict_t *other, cplane_t *plane, csurface_t *surf)
{
//If flag touched an invalid surface then return it immediately
	if (self->watertype & (CONTENTS_AUX | CONTENTS_LAVA | CONTENTS_SLIME))
	{
		CTF_Flag_Respawn(NULL, self->style, CTF_FLAG_RESPAWN_RESCUED);
		return;
	}
	if (surf)
	{
//If flag touches sky then respawn immediately
		if (surf->flags & SURF_SKY)
		{
			CTF_Flag_Respawn(NULL, self->style, CTF_FLAG_RESPAWN_RESCUED);
			return;
		}
	}
/*
//If flag touched a kill box then respawn immediately
	if (other)
	{
		if (strcmp(other->classname, "func_killbox") == 0)
		{
			CTF_Flag_Respawn(NULL, self->style, CTF_FLAG_RESPAWN_RESCUED);
			return;
		}
	}
*/
//Standard checks
	if (other->leader)
		other = other->leader;//Why is this here?

	if (!other->client)
		return;

	if(self->CTF_Flag.ReturnTimer > (FLAG_RETURN_DELAY - 10))//cant touch dropped flag for 1 second after drop
		return;

	if(other->client->pers.team == self->style)
	{
//same team - check if flag is at base - if not, return it
		if(CTF_Flag_At_Base(self) == true)
		{
			if (other->client->resp.PlayerData.Flag_State != CTF_DATA_FLAG_STATE_NULL)
			{
//Player is carrying enemy flag...only award 5 here since will get another 10 with rest of team
				gi.bprintf(PRINT_HIGH, "%s has captured the %s flag.\n", other->client->pers.netname, GetTeamName(other, TEAMNAME_OTHERTEAM));
				other->client->resp.PlayerData.Enemy_Flag_Total_Captures++;
				CTF_Game.team_score[other->client->pers.team - 1]++;
				CTF_Game.capture_time[other->client->pers.team - 1] = level.time + CTF_CAPTURE_FLASH_TIME;
				other->client->resp.score += CTF_POINTS_CAPTURE_ENEMY_FLAG;
			//Give other team members 10 points here
				AwardTeamBonus(other->client->pers.team);
//Check for home run or pickup
				if (other->client->resp.PlayerData.Flag_State & CTF_DATA_FLAG_STATE_CAPTURED)
				{
					other->client->resp.score += CTF_POINTS_HOMERUN_BONUS;//bonus for home run
					other->client->resp.PlayerData.Enemy_Flag_Stolen_Captures++;
				}
				else
					other->client->resp.PlayerData.Enemy_Flag_Pickup_Captures++;

				other->client->resp.PlayerData.Flag_State = CTF_DATA_FLAG_STATE_NULL;
				CTF_Flag_Respawn(NULL, other->client->pers.team, CTF_FLAG_RESPAWN_CAPTURED);
			}
		}
		else
		{
			other->client->resp.score += CTF_POINTS_RESCUE_OWN_FLAG;
			other->client->resp.PlayerData.Team_Flag_Rescued++;
			CTF_Flag_Respawn(NULL, self->style, CTF_FLAG_RESPAWN_RESCUED);
			gi.bprintf(PRINT_HIGH, "%s has rescued the %s flag.\n", other->client->pers.netname, GetTeamName(other, TEAMNAME_OWNTEAM));
		}
	}
	else
	{
//Opposing team
		if(CTF_Flag_At_Base(self) == true)
		{
//Player has picked up at base
			gi.sound (self, CHAN_NO_PHS_ADD + CHAN_VOICE, CTF_Game.Flag_Steal_Sound, 0.75f, ATTN_NONE, 0);
			gi.unlinkentity(self);
			G_FreeEdict(self);
			other->client->resp.PlayerData.Flag_State = CTF_DATA_FLAG_STATE_CAPTURED;
			other->client->resp.PlayerData.Enemy_Flag_Stolen++;
			other->client->resp.score += CTF_POINTS_TAKE_ENEMY_FLAG;
			gi.bprintf(PRINT_HIGH, "%s has stolen the %s flag.\n", other->client->pers.netname, GetTeamName(other, TEAMNAME_OTHERTEAM));
		}
		else
		{
//Player has picked up after it has been dropped
			gi.unlinkentity(self);
			G_FreeEdict(self);
			other->client->resp.PlayerData.Flag_State = CTF_DATA_FLAG_STATE_PICKEDUP;
			other->client->resp.PlayerData.Enemy_Flag_Pickups++;
			other->client->resp.score += CTF_POINTS_PICKUP_ENEMY_FLAG;
			gi.bprintf(PRINT_HIGH, "%s has picked up the %s flag.\n", other->client->pers.netname, GetTeamName(other, TEAMNAME_OTHERTEAM));
		}
	}
}

void ctf_flag_think (edict_t *self)
{
	if (self->s.frame < 48)
	{
		self->s.frame++;
	}
	else
	{
		self->s.frame = 0;
	}
	self->nextthink = level.time + FLAG_THINK_TIME;

	if(self->CTF_Flag.ReturnTimer > 0)
	{
		self->CTF_Flag.ReturnTimer--;
		if(self->CTF_Flag.ReturnTimer == 0)
			CTF_Flag_Respawn(NULL, self->style, CTF_FLAG_RESPAWN_RESCUED);
	}
}

void SP_CTF_Flag(edict_t *self)
{
//This routine should only be called at map load - use CTF_Flag_Respawn for any subsequent spawns
	int Team_Index;
	Flag_Initial_Spawn_Def* Flag_Initial;

	Team_Index = self->style - 1;

	if ((Team_Index < 0) || (Team_Index > 1))
	{
		gi.dprintf("%s has invalid style (should be 1 or 2) at %s\n", self->classname, vtos(self->s.origin));
		G_FreeEdict(self);
		return;
	}
//Could check for a flag with the incoming style already existing, but not going to bother.
//Style valid
	self->movetype = MOVETYPE_NONE;
	
//Save ent initial positions
	Flag_Initial = &Flag_Initial_Spawn[Team_Index];
	Flag_Initial->Exists = TRUE;
	VectorCopy(self->s.origin, Flag_Initial->Position);
	VectorCopy(self->s.angles, Flag_Initial->Angle);
//Set up the common parameters and link
	CTF_Flag_Set_Base_Spawn_Parameters(self, self->style);
}
