/*****************************************************************************

  b_filesys.c - File system

  Contains file opening routines, content processing, as well as any routine
  dedicated to writing or reading bot-related files (.bot, .way and .lst)

*****************************************************************************/

#include "g_local.h"
#if compileJACKBOT



  /**************************************************************************
     
		 Open files using proper mod directory

  **************************************************************************/
  FILE *fs_fileOpen(char *filename, char *mode)
		{
		cvar_t	*game_dir;
		FILE		*pFile;
		char		file_name[MAX_QPATH];

		// Get mod directory
		game_dir = gi.cvar("gamedir", "", 0);
		if (!strlen(game_dir->string))
			sprintf(file_name, "main\\%s", filename);
		else
			sprintf(file_name, "%s\\%s", game_dir->string, filename);

		// Get handle
		if (fopen_s(&pFile, file_name, mode))
			{
			if (!strncmp(mode, "w", 1))
				gi.dprintf("fs_fileOpen: unable to save %s\n", filename);
			else
				gi.dprintf("fs_fileOpen: unable to load %s\n", filename);
			return NULL;
			}
		return pFile;
		}



  /**************************************************************************
     
		 Read one line from a text file and clean it up

  **************************************************************************/
  qboolean fs_readLine(FILE *pIn, char *strSrc)
		{
		char  *lineFeed;
		int	  ofsSrc;
		int	  ofsDst = 0;
		int	  prevChr = 0x20;
		char  strDst[MAX_INFO_STRING];

		// Get line, remove line feed
		if (fgets(strSrc, MAX_INFO_STRING, pIn) == NULL)
			return false;
		if ((lineFeed = strchr(strSrc, '\n')) != NULL)
			*lineFeed = '\0';

		// Nothing to parse, bail
		if (!strlen(strSrc))
			return true;

		// Remove extra spacings, tabs and comments
		memset(strDst, '\0', sizeof(strDst));
		for (ofsSrc = 0; ofsSrc < strlen(strSrc); ofsSrc ++)
			{
			if (strSrc[ofsSrc] == 0x09)
				strSrc[ofsSrc] = 0x20;
			if ((prevChr == strSrc[ofsSrc]) && (strSrc[ofsSrc] == 0x20))
				continue;
			if ((prevChr == strSrc[ofsSrc]) && (strSrc[ofsSrc] == 0x2F))
				{
				ofsDst -= 1;
				break;
				}
			strDst[ofsDst] = strSrc[ofsSrc];
			prevChr = strDst[ofsDst];
			ofsDst++;
			}
		strDst[ofsDst] = 0x00;

		// Remove excess last space, if any
		if (strlen(strDst) > 1)
			if (strDst[ofsDst - 1] == 0x20)
				strDst[ofsDst - 1] = 0x00;

		// Copy string
		strcpy(strSrc, strDst);
		return true;
		}



  /**************************************************************************
     
		 Split a string in two

  **************************************************************************/
  qboolean fs_splitLine(char *src, char *key, char *value)
		{
		char  *ptrStart;

		// No space, bail
		ptrStart = strchr(src, 0x20);
		if (!ptrStart)
			return false;

		// Get value (right part of the string)
		strncpy(value, ptrStart + 1, strlen(ptrStart) - 1);
		value[strlen(ptrStart) - 1] = 0x00;

		// Get key (left part of the string)
		strncpy(key, src, strlen(src) - strlen(ptrStart));
		key[strlen(src) - strlen(ptrStart)] = 0x00;

		return true;
		}



  /**************************************************************************
  
	   BOTS: save script (bots/*.bot)
  
	**************************************************************************/
  void fs_saveBotScript(char *name)
		{
		edict_t	  **botList = 0;
		edict_t	  *bot;
		int				botCount;
		FILE			*pOut;
		int				i;
		char			filename[MAX_QPATH];
		char			msg[MAX_INFO_STRING];
		field_t	  *f;
		char			spc[MAX_INFO_STRING];
		int				maxLen = 0;
		char			*spcPtr;

		// Bot select
		botCount = botMisc_CatchBot(name, &botList);
		if (!botCount)
			return;

		// Get alignment
		memset (spc, 0x20, MAX_INFO_STRING - 1);
		for (f = jb_Script; f->name; f++)
			{
			if (f->ofs == 0)
				continue;
			if (strlen(f->name) > maxLen)
				maxLen = strlen(f->name);
			}
		maxLen += 2;

		// Write scripts
		for (i = 0; i < botCount; i++)
			{
			bot = botList[i];

			// File open
			sprintf(filename, "bots\\%s.bot", bot->client->pers.netname);
			pOut = fs_fileOpen(filename, "w");
			if (!pOut)
				continue;

			// Write file
			fprintf(pOut, "//===========================================================================\n//\n");
			fprintf(pOut, "// Bot script file for %s\n", bot->client->pers.netname);
			fprintf(pOut, "//\n//===========================================================================\n");
			for (f = jb_Script; f->name; f++)
				{
				memset (msg, 0, MAX_INFO_STRING);
				botMisc_attribute(&bot->botInfo->def, f->name, msg);
				if (strlen(msg))
					{
					spcPtr = spc;
					if ((maxLen - strlen(f->name)) < MAX_INFO_STRING)
						spcPtr += MAX_INFO_STRING - (maxLen - strlen(f->name));
					fprintf(pOut, "  %s%s%s\n", f->name, spcPtr, msg);
					}
				else
					fprintf(pOut, "\n// %s\n", f->name);
				}

			// Close file and free memory
			fclose(pOut);
			gi.dprintf("Wrote %s\n", filename);
			}
		}



  /**************************************************************************
  
	   BOTS: save list (maps/*.lst)
  
	**************************************************************************/
  void fs_saveBotList()
		{
		edict_t	  **botList = 0;
		edict_t	  *bot;
		int				botCount;
		FILE			*pOut;
		int				i, cnt;
		char			filename[MAX_QPATH];
	
		// Bot select
		botCount = botMisc_CatchBot("all", &botList);
		if (!botCount)
			return;

		// File open
		sprintf(filename, "maps\\%s.lst", level.mapname);
		pOut = fs_fileOpen(filename, "w");
		if (!pOut)
			{
			free(botList);
			return;
			}

		// File write
		fprintf(pOut, "//===========================================================================\n//\n");
		fprintf(pOut, "// Bot list for %s.bsp\n", level.mapname);
		fprintf(pOut, "//\n//===========================================================================\n\n");

		cnt = 0;
		for (i = 0; i < botCount; i++)
			{
			bot = botList[i];
			if (bot->client->pers.team != TEAM_1)
				continue;
			if (cnt == 0)
				fprintf(pOut, "// TEAM 1: Dragon\n\n");
			fprintf(pOut, "BOT_TEAM1 %s\n", bot->client->pers.netname);
			}

		cnt = 0;
		for (i = 0; i < botCount; i++)
			{
			bot = botList[i];
			if (bot->client->pers.team != TEAM_2)
				continue;
			if (cnt == 0)
				fprintf(pOut, "// TEAM 2: Poison\n\n");
			fprintf(pOut, "BOT_TEAM2 %s\n", bot->client->pers.netname);
			}

		cnt = 0;
		for (i = 0; i < botCount; i++)
			{
			bot = botList[i];
			if (bot->client->pers.team != TEAM_NONE)
				continue;
			if (cnt == 0)
				fprintf(pOut, "// DEATHMATCH\n\n");
			fprintf(pOut, "BOT_DEATHMATCH %s\n", bot->client->pers.netname);
			}
		// Close file and free memory
		fclose(pOut);
		gi.dprintf("Wrote %s\n", filename);
		}

  
	
	/**************************************************************************
    
		BOTS: load list (maps/*.lst)
  
	**************************************************************************/
  void fs_loadBotList()
		{
		FILE		*pIn;
		char		filename[MAX_QPATH];
		char		fileLine[MAX_INFO_STRING];
		char		lineKey[MAX_INFO_STRING];
		char		lineValue[MAX_INFO_STRING];

		// Open file
		sprintf(filename, "maps\\%s.lst", level.mapname);
		pIn = fs_fileOpen(filename, "r");
		if (!pIn)
			return;
		gi.dprintf("LoadList: Loading %s...\n", filename);

		// Read the whole file, one line at a time (teamplay)
		if (teamplay->value)
			{
			while (fs_readLine(pIn, fileLine))
				{
				if (fs_splitLine(fileLine, lineKey, lineValue) == false)
					continue;
				if (!Q_stricmp("BOT_TEAM1", lineKey))
					fs_loadBotScript(lineValue, TEAM_1);
				else if (!Q_stricmp("BOT_TEAM2", lineKey))
					fs_loadBotScript(lineValue, TEAM_2);
				}
			}

		// Read the whole file, one line at a time (free for all)
		else
			{
			while (fs_readLine(pIn, fileLine))
				{
				if (fs_splitLine(fileLine, lineKey, lineValue) == false)
					continue;
				if (!Q_stricmp("BOT_DEATHMATCH", lineKey))
					fs_loadBotScript(lineValue, TEAM_NONE);
				}
			}

		// Close
		fclose(pIn);
		}

  
	/*************************************************************************

		Bot behavior, skills and needed info is located in an external array
		to reduce memory usage (no need to store 1Kb of bot-related info for
		each entity present on the map)

	*************************************************************************/
	botinfo_t *freeBotRegistery()
		{
		botDef_t	*def;
		int				i;
		#if (compileTEST)
		char			tmp[MAX_INFO_STRING];
		#endif
		for (i = 0; i < MAX_CLIENTS; i++)
			{
			if (jb_BotInfo[i].inUse)
				continue;
			// Reset all
			memset(&jb_BotInfo[i],  0x00, sizeof(jb_BotInfo[i]));
			def = &jb_BotInfo[i].def;
			// Default style
			#if (compileTEST)
			sprintf(tmp, "bot%i", i);
			def->ofsName			= fs_stringBankAppend(&jb_StringBank, &jb_StringBankSize, tmp);
			def->ofsSkin			= fs_stringBankAppend(&jb_StringBank, &jb_StringBankSize, "male_thug/009 019 017");
			def->ofsSkinTeam1	= fs_stringBankAppend(&jb_StringBank, &jb_StringBankSize, "male_thug/009 056 032");
			def->ofsSkinTeam2	= fs_stringBankAppend(&jb_StringBank, &jb_StringBankSize, "male_thug/009 035 030");
			def->ofsExtras		= fs_stringBankAppend(&jb_StringBank, &jb_StringBankSize, "0");
			def->ofsHand			= fs_stringBankAppend(&jb_StringBank, &jb_StringBankSize, "2");
			#else
			sprintf(def->name,			"bot%i", i);
			strcpy(def->skin,				"male_thug/009 019 017");
			strcpy(def->skinteam1,	"male_thug/009 056 032");
			strcpy(def->skinteam2,	"male_thug/009 035 030");
			strcpy(def->extras,			"0");
			strcpy(def->hand,				"2");
			#endif
			// Default skills
			def->accuracy					= 0.50;
			def->jumper						= 0.20;
			def->croucher					= 0.03;
			def->spaceaware				= 1.00;
			def->preserve					= 0.50;
			def->reloader					= 1.00;
			def->camper						= 0.30;
			def->reactiontime			= 3.10;
			def->dodge						= 0.80;
			def->jittery					= 0.10;
			def->chatter					= 0.75;
			def->viewangle				= 120;
			def->viewdistance			= 2048;
			def->flags						= 0;
			// Default chat messages
			#if (compileTEST)
			def->ofsChatGameover			= fs_stringBankAppend(&jb_StringBank, &jb_StringBankSize, "GG");
			def->ofsChatGameoverSmug	= def->ofsChatGameover;
			def->ofsChatGameoverAngry	= def->ofsChatGameover;
			def->ofsChatBagmanSpotted	= fs_stringBankAppend(&jb_StringBank, &jb_StringBankSize, "Enemy spotted!");
			def->ofsChatGamejoin			= fs_stringBankAppend(&jb_StringBank, &jb_StringBankSize, "Hi!");
			#else
			strcpy(def->c_gameover_smug,	"");
			strcpy(def->c_gameover_angry, "");
			strcpy(def->c_gameover,				"");
			strcpy(def->c_bagman_spotted, "");
			strcpy(def->c_gamejoin,				"");
			strcpy(def->c_respawn,				"");
			#endif
			// Default best weapons
			memset(def->rankWeight, -1, sizeof(float) * MAX_WEAPONS);
			memset(def->rankOrder,  -1, sizeof(float) * MAX_WEAPONS);
			def->rankWeight[0]		= 0.02;
			def->rankWeight[1]		= 0.05;
			def->rankWeight[2]		= 0.15;
			def->rankWeight[3]		= 0.40;
			def->rankWeight[4]		= 0.50;
			def->rankWeight[5]		= 1.00;
			def->rankWeight[6]		= 0.70;
			def->rankWeight[7]		= 0.90;
			def->rankWeight[8]		= 0.60;
			// Return pointer (while it's initialized, it's still not in use)
			return &jb_BotInfo[i];
			}
		return NULL;
		}


	/**************************************************************************
    
		BOTS: load script (bots/*.bot)

  **************************************************************************/
  void fs_loadBotScript(char *name, int team)
		{
		FILE			*pIn;
		char			filename[MAX_QPATH];
		char			fileLine[MAX_INFO_STRING];
		char			lineKey[MAX_INFO_STRING];
		char			lineValue[MAX_INFO_STRING];
		botinfo_t	*botInfo;
		int				lineNum = 0;

		if (!name)
			return;

		// Open file
		sprintf(filename, "bots\\%s.bot", name);
		pIn = fs_fileOpen(filename, "r");
		if (!pIn)
			return;

		// Get registery
		botInfo = freeBotRegistery();
		if (!botInfo)
			{
			gi.dprintf("fs_loadBotScript: unable to register bot info\n");
			return;
			}

		// Read the whole file, one line at a time
		while (fs_readLine(pIn, fileLine))
			{
			lineNum++;
			if (fs_splitLine(fileLine, lineKey, lineValue) == false)
				continue;
			if (!botMisc_attribute(&(botInfo->def), lineKey, lineValue))
				gi.dprintf("Attribute unknown \"%s\" (line %i)\n", lineKey, lineNum);
			}

		// Close
		fclose(pIn);

		// Spawn bot
		botMisc_rankWeapons(&(botInfo->def));
		BotSpawn(botInfo, team);
		}



	/**************************************************************

	  NODES: support for kraze routes (maps/*.kpr).
		This routine is called by fs_loadNode if it failed loading
		the appropriate *.way file. If the *.kpr file is loaded
		successfully, it is saved again to the appropriate .way

	**************************************************************/
	qboolean fs_loadNodesKraze()
		{
		FILE		*pIn;
		char		filename[MAX_QPATH];
		int			i, j;

		INT32		version;
		INT32		numNodes;
		INT32		numItems;
		vec3_t	nodeOrg;
		INT32		nodeType;
		INT32		nodeIndex;
		INT32		itemIndex;
		INT32		linkedNode;
		edict_t *linkEnt;
		INT16		nodeLink;

		sprintf(filename, "maps\\%s.kpr", level.mapname);
		pIn = fs_fileOpen(filename, "rb");
		if (!pIn)
			return true;

		// Get version, numNodes and numItesm
		fread(&version, sizeof(version), 1, pIn);
		fread(&numNodes, sizeof(numNodes), 1, pIn);
		fread(&numItems, sizeof(numItems), 1, pIn);
		if ((version != 3) || ((numNodes < 1) || (numNodes >= BOTNODE_MAX)))
			{
			fclose(pIn);
			return true;
			}

		// NODES //
		for (i = 0; i < numNodes; i++)
			{
			// Get Kraze coordinates and type
			fread(&nodeOrg,		sizeof(nodeOrg),		1, pIn);
			fread(&nodeType,	sizeof(nodeType),		1, pIn);
			fread(&nodeIndex,	sizeof(nodeIndex),	1, pIn);
			fseek(pIn, 96L, SEEK_CUR);
			// Re-assign coordinates
			VectorCopy(nodeOrg, jb_Node[nodeIndex].origin);
			// Re-assign type
			jb_Node[nodeIndex].type = getNodeTypeByKrazeValue(nodeType);
			if (!jb_Node[nodeIndex].type)
				{
				jb_Node[nodeIndex].type = getNodeTypeByValue(nMove);
				gi.dprintf("KRAZE: Node[%i] at %s of type %i converted to %s.\n", nodeIndex, vtos(nodeOrg), nodeType, jb_Node[nodeIndex].type->name);
				}
			}
		jb_NumNodes = numNodes;

		// ROUTES //
		for (i = 0; i < numNodes; i++)
			{
			for (j = 0; j < numNodes; j++)
				{
				fread(&nodeLink, sizeof(nodeLink), 1, pIn);
				jb_PathTable[i][j] = -1;
				if ((nodeLink >= 0) && (nodeLink < numNodes))
					{
					if (jb_Node[nodeLink].type)
						jb_PathTable[i][j] = nodeLink;
					}
				}
			}

		// ITEMS //
		for (i = 0; i < numItems; i++)
			{
			fread(&itemIndex, sizeof(itemIndex), 1, pIn);
			fseek(pIn, 8L, SEEK_CUR);
			fread(&linkedNode, sizeof(linkedNode), 1, pIn);
			nodeLink = (short)linkedNode;
			
			if ((nodeLink < 0) || (nodeLink >= numNodes)) // "Node[%i] can't be tied to entity: out of range!\n", nodeLink
				continue;
			
			if (!jb_Node[nodeLink].type) // "Node[%i] at %s can't be tied to entity %i, undefined.\n", nodeLink, vtos(jb_Node[nodeLink].origin), itemIndex
				continue;
			
			if (!jb_Node[nodeLink].type->linksEntity) // "Node[%i] at %s can't be tied to entity %i, invalid type.\n", nodeLink, vtos(jb_Node[nodeLink].origin), itemIndex
				continue;

			// find closest entity to that origin
			linkEnt = getClosestEntity(NULL, jb_Node[nodeLink].origin, 64);
			if (!linkEnt)
				{
				jb_Node[nodeLink].type = getNodeTypeByValue(nMove);
				gi.dprintf("KRAZE: Node[%i] at %s, unable to find entity index %i.\n", nodeLink, vtos(jb_Node[nodeLink].origin), itemIndex);
				continue;
				}

			// now, check if this entity is part of itemlist[]
			j = botMisc_FindItemIndexByClassname(linkEnt->classname);
			jb_ItemTable[jb_NumItems].item = j;						// index in itemlist[] (-1 if not part of the itemlist)
			jb_ItemTable[jb_NumItems].node = nodeLink;		// link item to jb_Node[i]
			jb_ItemTable[jb_NumItems].ent	 = linkEnt;			// pointer to entity

			// not present in itemlist[], could be another valuable entity
			if (j == -1)
				{
				if (!Q_stricmp(linkEnt->classname, "func_plat"))
					jb_Node[nodeLink].type = getNodeTypeByValue(nElevator);
				else if (!Q_stricmp(linkEnt->classname, "misc_teleporter_dest") || !Q_stricmp(linkEnt->classname, "misc_teleporter"))
					jb_Node[nodeLink].type = getNodeTypeByValue(nTeleporter);
				if (jb_Node[nodeLink].type->value == nItem)
					{
					jb_Node[nodeLink].type = getNodeTypeByValue(nMove);
					continue;
					}
				}

			jb_NumItems++;
			}

		// REMOVE EXCESS NODES //
		node_removeExcessNodes();

		// DONE //
		fclose(pIn);
		gi.dprintf("Loaded %s.\n", filename);
		fs_saveNodes(false);
		return false;
		}

	

  /**************************************************************

     NODES: load file (maps/*.way)

  **************************************************************/
  void fs_loadNodes()
		{
		// routine stuff
		FILE								*pIn;
		char								filename[MAX_QPATH];
		char								padding[16];
		int									i, j;

		// file stuff
		unsigned int				magic;				// 4 - Signature
		unsigned short int	version;			// 2 - Version
		unsigned short int	botversion;		// 2 - Bot version
		unsigned short int	revision;			// 2 - Roadmap revision number
		unsigned short int	author;				// 2 - Author netname
		unsigned short int	bankLength;		// 2 - String bank size
		unsigned short int	numChunks;		// 2 - Number of chunks
		byte								*bankBuffer;	// ? - String bank
		fs_Chunk_t					chunk[6];			// ? - Chunks

		// SETUP CHUNKS //
		memset(padding, 0x00, 16);
		memset(chunk, 0x00, sizeof(chunk));
		
		// OPEN FILE //
		sprintf(filename, "maps\\%s.way", level.mapname);
		pIn = fs_fileOpen(filename, "rb");
		if (!pIn)
			{
			if (!fs_loadNodesKraze()) // attempt to load KPL file instead
				return;
			edit_BuildItemNodeTable();
			return;
			}

		// SIZE CHECK //
		fseek(pIn, 0L, SEEK_END);
		i = ftell(pIn);
		fseek(pIn, 0L, SEEK_SET);
		if ((i < 32) || (i % 16))
			{
			fclose(pIn);
			gi.dprintf("Cannot load %s, file corrupted or invalid.\n", filename);
			if (!fs_loadNodesKraze()) // attempt to load KPL file instead
				return;
			edit_BuildItemNodeTable();
			return;
			}

		// READ HEADER //
		fread(&magic,				sizeof(magic),			1, pIn);	// 4 - signature
		fread(&version,			sizeof(version),		1, pIn);	// 2 - version
		fread(&botversion,	sizeof(botversion),	1, pIn);	// 2 - bot version
		fread(&revision,		sizeof(revision),		1, pIn);	// 2 - revision number
		fread(&author,			sizeof(author),			1, pIn);	// 2 - author netname
		fread(&bankLength,	sizeof(bankLength),	1, pIn);	// 2 - string bank size, in bytes
		fread(&numChunks,		sizeof(numChunks),	1, pIn);	// 2 - number of chunks

		// CHECK //
		if ((magic != 0x74426B4A) || (version != 5) || (!bankLength) || (!numChunks))
			{
			fclose(pIn);
			if (!fs_loadNodesKraze()) // attempt to load KPL file instead
				return;
			if (magic != 0x74426B4A)
				gi.dprintf("Cannot load %s, wrong signature.\n", filename);
			if (version != 5)
				gi.dprintf("Cannot load %s, is version %i (expected 5).\n", filename, version);
			if (!bankLength)
				gi.dprintf("Cannot load %s, string bank is null.\n", filename);
			if (!numChunks)
				gi.dprintf("Cannot load %s, no chunk in the directory.\n", filename);
			edit_BuildItemNodeTable();
			return;
			}

		// READ STRING BANK //
		bankBuffer = (byte *)malloc(bankLength);
		memset(bankBuffer, 0x00, bankLength);
		fread(bankBuffer, bankLength, 1, pIn);
		if (bankLength % 16)
			fread(padding, 16 - (bankLength % 16), 1, pIn);

		// READ CHUNK DIRECTORY //
		for (i = 0; i < numChunks; i++)
			{
			fread(&chunk[i].label,			sizeof(chunk[i].label),				1, pIn); // 2
			fread(&chunk[i].version,		sizeof(chunk[i].version),			1, pIn); // 2
			fread(&chunk[i].dataLength,	sizeof(chunk[i].dataLength),	1, pIn); // 4, length of data (may or not be compressed)
			fread(&chunk[i].rleLength,	sizeof(chunk[i].rleLength),		1, pIn); // 4, length of RLE INSTRUCTIONS
			fread(&chunk[i].wrLength,		sizeof(chunk[i].wrLength),		1, pIn); // 4, length of WR INSTRUCTIONS
			j = (chunk[i].rleLength + chunk[i].wrLength + chunk[i].dataLength);
			if (j % 16)
				chunk[i].padding = 16 - (j % 16);
			else
				chunk[i].padding = 0;
			}

		// READ CHUNK //
		for (i = 0; i < numChunks; i++)
			{
			if (chunk[i].rleLength)
				{
				chunk[i].rleBuffer = (byte *)malloc(chunk[i].rleLength);
				fread(chunk[i].rleBuffer, chunk[i].rleLength, 1, pIn);
				}
			if (chunk[i].wrLength)
				{
				chunk[i].wrBuffer = (byte *)malloc(chunk[i].wrLength);
				fread(chunk[i].wrBuffer, chunk[i].wrLength, 1, pIn);
				}
			if (chunk[i].dataLength)
				{
				chunk[i].dataBuffer = (short *)malloc(chunk[i].dataLength);
				fread(chunk[i].dataBuffer, chunk[i].dataLength, 1, pIn);
				}
			if (chunk[i].padding)
				{
				memset(padding, 0x00, 16);
				fread(padding, chunk[i].padding, 1, pIn);
				}
			fs_ChunkApply(chunk[i], bankBuffer, bankLength);
			fs_ChunkDestroy(chunk[i]);
			}
		
		// DONE //
		fclose(pIn);

		jb_NumRevs = revision;
		gi.dprintf("Loaded %s (%s, REV: %i)\n", filename, &(((char *)bankBuffer)[author]), revision);
		if (botversion > jb_BotVersion)
			gi.dprintf("**** Roadbook for botVersion %i, current version is %i ****\n", botversion, jb_BotVersion);

		free(bankBuffer);
		}



  /**************************************************************

    Save nodes (maps/*.way)

  **************************************************************/
  void fs_saveNodes(qboolean quicksave)
		{
		// routine stuff
		FILE								*pOut;
		char								filename[MAX_QPATH];
		char								padding[16];
		int									i;
		unsigned int				initialSize	= 0;	// size of data, uncompressed
		unsigned int				actualSize  = 0;	// size of data, compressed

		// file stuff
		unsigned int				magic				= 0x74426B4A;			// 4 - Signature
		unsigned short int	version			= 5;							// 2 - Version
		unsigned short int	botversion	= jb_BotVersion;	// 2 - Bot version
		unsigned short int	revision		= jb_NumRevs + 1;	// 2 - Roadmap revision number
		unsigned short int	author  		= 0;							// 2 - Author netname
		unsigned short int	bankLength	= 0;							// 2 - String bank size
		unsigned short int	numChunks		= 0;							// 2 - Number of chunks
		byte								*bankBuffer;									// ? - String bank
		fs_Chunk_t					chunk[6];											// ? - Chunks

		// OPEN FILE //
		if (!quicksave)
			{
			sprintf(filename, "maps\\%s.way", level.mapname);
			jb_NumRevs = revision;
			}
		else
			sprintf(filename, "maps\\%s.tmp", level.mapname);
		pOut = fs_fileOpen(filename, "wb");
		if (!pOut)
			return;

		// SETUP STRING BANK //
		fs_stringBankSetup(&bankBuffer, &bankLength, true);
		author = fs_stringBankAppend(&bankBuffer, &bankLength, jb_Player[0]->client->pers.netname);

		// SETUP CHUNKS //
		memset(padding, 0x00, 16);
		memset(chunk, 0x00, sizeof(chunk));
		if (jb_NumNodes)
			{
			chunk[numChunks] = fs_ChunkSetup("NodeLst", &bankBuffer, &bankLength, quicksave); numChunks++; // Put this one first for convenience's sake
			chunk[numChunks] = fs_ChunkSetup("RMapDef", &bankBuffer, &bankLength, quicksave); numChunks++;
			}
		if (jb_NumJumps)
			{
			chunk[numChunks] = fs_ChunkSetup("LinkJmp", &bankBuffer, &bankLength, quicksave);
			numChunks++;
			}
		if (jb_NumDucks)
			{
			chunk[numChunks] = fs_ChunkSetup("LinkDuk", &bankBuffer, &bankLength, quicksave);
			numChunks++;
			}
		if (jb_NumLinks)
			{
			chunk[numChunks] = fs_ChunkSetup("LinkUsr", &bankBuffer, &bankLength, quicksave);
			numChunks++;
			}

		// WRITE HEADER //
		fwrite(&magic,				sizeof(magic),				1, pOut);	// 4 - signature
		fwrite(&version,			sizeof(version),			1, pOut); // 2 - version
		fwrite(&botversion,		sizeof(botversion),		1, pOut); // 2 - bot version
		fwrite(&revision,			sizeof(revision),			1, pOut); // 2 - revision number
		fwrite(&author,				sizeof(author),				1, pOut); // 2 - author netname
		fwrite(&bankLength,		sizeof(bankLength),		1, pOut); // 2 - string bank size, in bytes
		fwrite(&numChunks,		sizeof(numChunks),		1, pOut); // 2 - number of chunks
		fwrite(bankBuffer, bankLength, 1, pOut);
		if (bankLength % 16)
			fwrite(padding, 16 - (bankLength % 16), 1, pOut);
		for (i = 0; i < numChunks; i++)
			{
			fwrite(&chunk[i].label,				sizeof(chunk[i].label),				1, pOut);	// 2 - chunk type, offset in the string bank
			fwrite(&chunk[i].version,			sizeof(chunk[i].version),			1, pOut);	// 2 - chunk version, should always be 0
			fwrite(&chunk[i].dataLength,	sizeof(chunk[i].dataLength),	1, pOut);	// 4 - length of useful data, in bytes
			fwrite(&chunk[i].rleLength,		sizeof(chunk[i].rleLength),		1, pOut);	// 4 - length of compression intruction (RLE), in bytes
			fwrite(&chunk[i].wrLength,		sizeof(chunk[i].wrLength),		1, pOut);	// 4 - length of compression intruction (WR), in bytes
			}

		// WRITE CHUNKS (DESTROY AFTER USAGE) //
		for (i = 0; i < numChunks; i++)
			{
			if (chunk[i].rleLength)
				fwrite(chunk[i].rleBuffer, chunk[i].rleLength, 1, pOut);
			if (chunk[i].wrLength)
				fwrite(chunk[i].wrBuffer, chunk[i].wrLength, 1, pOut);
			if (chunk[i].dataLength)
				fwrite(chunk[i].dataBuffer, chunk[i].dataLength, 1, pOut);
			if (chunk[i].padding)
				fwrite(padding,	chunk[i].padding, 1, pOut);
			initialSize += chunk[i].initial;
			actualSize  += chunk[i].dataLength + chunk[i].wrLength + chunk[i].rleLength;
			fs_ChunkDestroy(chunk[i]);
			}

		// DONE //
		fclose(pOut);
		free(bankBuffer);

		// Summary
		if (quicksave)
			if ((int)bot_nodeautosave->value < 30)
				gi.dprintf("Wrote %s, next save in 30 seconds...\n", filename);
			else
				gi.dprintf("Wrote %s, next save in %i seconds...\n", filename, (int)bot_nodeautosave->value);
		else
			{
			if (actualSize < initialSize)
				gi.dprintf("Wrote %s (crunched %i bytes)\n", filename, initialSize - actualSize);
			else
				gi.dprintf("Wrote %s\n", filename);
			}
		}



	/************************************************************

		Free memory allocated for the provided chunk

	************************************************************/
	void fs_ChunkDestroy(fs_Chunk_t chunk)
		{
		if (chunk.rleBuffer)
			{
			free(chunk.rleBuffer);
			chunk.rleBuffer = 0;
			}
		if (chunk.wrBuffer)
			{
			free(chunk.wrBuffer);
			chunk.wrBuffer = 0;
			}
		if (chunk.dataBuffer)
			{
			free(chunk.dataBuffer);
			chunk.dataBuffer = 0;
			}
		chunk.initial			= 0;
		chunk.padding			= 0;
		chunk.rleLength		= 0;
		chunk.wrLength		= 0;
		chunk.dataLength	= 0;
		chunk.label				= 0;
		chunk.version			= 0;
		}


	
	/************************************************************

		Apply specified chunk

	************************************************************/
	void fs_ChunkApply(fs_Chunk_t chunk, byte *strBuffer, unsigned short int strLength)
		{
		int							i, j;
		char						*label, *nodeEntity, *nodeLabel, *moveType;
		edict_t					*linkEnt = NULL;
		nodeType_t			*initType;
		trace_t					tr;
		vec3_t					v;

		label = &(((char *)strBuffer)[chunk.label]);
		
		// EXPAND //
		if (chunk.wrLength)
			{
			if (wr_Expand(&chunk.dataBuffer, &chunk.dataLength, &chunk.wrBuffer, &chunk.wrLength))
				{
				gi.dprintf("fs_ChunkApply: couldn't apply chunk %s\n", label);
				return;
				}
			}
		if (chunk.rleLength)
			{
			if (rle_Expand(&chunk.dataBuffer, &chunk.dataLength, &chunk.rleBuffer, &chunk.rleLength))
				{
				gi.dprintf("fs_ChunkApply: couldn't apply chunk %s\n", label);
				return;
				}
			}

		// DATA BUFFER // Default roadbook //
		if (!strcmp(label, "RMapDef") && (chunk.version == 0))
			{
			short *ptr = (short *)chunk.dataBuffer;
			if (chunk.dataLength != jb_NumNodes * jb_NumNodes * 2)
				{
				gi.dprintf("fs_ChunkApply: expected %i bytes of data, received %i for %s\n", jb_NumNodes * jb_NumNodes * 2, chunk.dataLength, label);
				return;
				}
			for (i = 0; i < jb_NumNodes; i++)
				for (j = 0; j < jb_NumNodes; j++)
					jb_PathTable[i][j] = ptr[(i * jb_NumNodes) + j];	// 2 - Proxy to reach [j] from [i]
			}

		// DATA BUFFER // Node list //
		else if (!strcmp(label, "NodeLst") && (chunk.version == 0))
			{
			fs_NodeSave_t *ptr = (fs_NodeSave_t *)chunk.dataBuffer;
			if (chunk.dataLength % (sizeof(fs_NodeSave_t)))
				{
				gi.dprintf("Chunk \"%s\" is corrupted.\n", label);
				return;
				}
			jb_NumNodes = (chunk.dataLength / (sizeof(fs_NodeSave_t)));
			for (i = 0; i < jb_NumNodes; i++)
				{
				nodeLabel		= &(((char *)strBuffer)[ptr[i].nodeLabel]);		// 2 - Node name
				nodeEntity	= &(((char *)strBuffer)[ptr[i].entityLabel]);	// 2 - Entity name
				jb_Node[i].origin[0] = ptr[i].x;													// 4 - x coordinate
				jb_Node[i].origin[1] = ptr[i].y;													// 4 - x coordinate
				jb_Node[i].origin[2] = ptr[i].z;													// 4 - x coordinate
				// Apply
				jb_Node[i].type = getNodeTypeByName(nodeLabel);
				if (!jb_Node[i].type)
					{
					jb_Node[i].type = getNodeTypeByValue(nMove);
					gi.dprintf("Node[%i] at %s has type %s, converted node to %s\n", i, vtos(jb_Node[i].origin), nodeLabel, jb_Node[i].type->name);
					}
				if ((jb_Node[i].type->linksEntity) && (nodeEntity))
					{
					if (jb_Node[i].type->value == nElevator)
						{
						v[0] = jb_Node[i].origin[0];
						v[1] = jb_Node[i].origin[1];
						v[2] = jb_Node[i].origin[2] - 4096;
						tr = gi.trace(jb_Node[i].origin, NULL, NULL, v, NULL, MASK_SOLID);
						if (tr.fraction < 1.00)
							linkEnt = tr.ent;
						}
					else
						linkEnt = getClosestEntity(nodeEntity, jb_Node[i].origin, 64);
					if (!linkEnt)
						{
						initType = jb_Node[i].type;
						jb_Node[i].type = getNodeTypeByValue(nMove);
						gi.dprintf("Node[%i] at %s type %s has lost \"%s\", converted to %s\n", i, vtos(jb_Node[i].origin), initType->name, nodeEntity, jb_Node[i].type->name);
						}
					else
						{
						j = botMisc_FindItemIndexByClassname(nodeEntity);
						jb_ItemTable[jb_NumItems].item = j;				// index in itemlist[]
						jb_ItemTable[jb_NumItems].node = i;				// link item to jb_Node[i]
						jb_ItemTable[jb_NumItems].ent	 = linkEnt;	// pointer to entity
						jb_ItemTable[jb_NumItems].turf = getEntityTerritory(linkEnt);	// get the turf where this item is located
						jb_NumItems++;
						}
					}
				}
			}

		// DATA BUFFER // Jump links //
		else if (!strcmp(label, "LinkJmp") && (chunk.version == 0))
			{
			nodePair_t *ptr = (nodePair_t *)(chunk.dataBuffer);
			if (chunk.dataLength % (sizeof(nodePair_t)))
				{
				gi.dprintf("Chunk \"%s\" is corrupted.\n", label);
				return;
				}
			jb_NumJumps = (chunk.dataLength / (sizeof(nodePair_t)));
			for (i = 0; i < jb_NumJumps; i++)
				{
				jb_JumpTable[i].from	= ptr[i].from;
				jb_JumpTable[i].to		= ptr[i].to;
				}
			}

		// DATA BUFFER // Duck Links //
		else if (!strcmp(label, "LinkDuk") && (chunk.version == 0))
			{
			nodePair_t *ptr = (nodePair_t *)(chunk.dataBuffer);
			if (chunk.dataLength % (sizeof(nodePair_t)))
				{
				gi.dprintf("Chunk \"%s\" is corrupted.\n", label);
				return;
				}
			jb_NumDucks = (chunk.dataLength / (sizeof(nodePair_t)));
			for (i = 0; i < jb_NumDucks; i++)
				{
				jb_DuckTable[i].from	= ptr[i].from;
				jb_DuckTable[i].to		= ptr[i].to;
				}
			}

		// DATA BUFFER // User Links //
		else if (!strcmp(label, "LinkUsr") && (chunk.version == 0))
			{
			nodePair2_t *ptr = (nodePair2_t *)(chunk.dataBuffer);
			if (chunk.dataLength % (sizeof(nodePair2_t)))
				{
				gi.dprintf("Chunk \"%s\" is corrupted.\n", label);
				return;
				}
			jb_NumLinks = (chunk.dataLength / (sizeof(nodePair2_t)));
			for (i = 0; i < jb_NumLinks; i++)
				{
				jb_LinkTable[i].from	  = ptr[i].from;
				jb_LinkTable[i].to		  = ptr[i].to;

				moveType = &(((char *)strBuffer)[ptr[i].vehicle]);
				if (!Q_stricmp(moveType, "jump"))
					jb_LinkTable[i].vehicle = VEHICLE_JUMP;
				else if (!Q_stricmp(moveType, "duck"))
					jb_LinkTable[i].vehicle = VEHICLE_CROUCH;
				else
					{
					if (Q_stricmp(moveType, "walk"))
						gi.dprintf("User link number %i has move type %s, converted to walk\n", i, moveType);
					jb_LinkTable[i].vehicle = VEHICLE_WALK;
					}
				}
			}

		// No idea what this is //
		else
			{
			gi.dprintf("Chunk \"%s\" (v%i) unknown.\n", label, chunk.version);
			return;
			}
		}



	/************************************************************

		Produce the requested chunk

	************************************************************/
	fs_Chunk_t fs_ChunkSetup(char *label, byte **strBuffer, unsigned short int *strLength, qboolean quicksave)
		{
		int					i, j, k;
		byte				*baseBuffer;
		fs_Chunk_t	chunk;
		
		// RESET CHUNK
		memset(&chunk, 0x00, sizeof(fs_Chunk_t));

		// DATA BUFFER // Default roadbook //
		if (!strcmp(label, "RMapDef"))
			{
			chunk.initial = jb_NumNodes * jb_NumNodes * sizeof(short);
			chunk.dataLength = chunk.initial;
			chunk.dataBuffer = (short *)malloc(chunk.dataLength);
			chunk.version = 0;
			for (i = 0; i < jb_NumNodes; i++)
				{
				k = i * jb_NumNodes;
				for (j = 0; j < jb_NumNodes; j++)
					chunk.dataBuffer[k + j] = jb_PathTable[i][j];	// 2 - Proxy to reach [j] from [i]
				}
			}

		// DATA BUFFER // Node list //
		else if (!strcmp(label, "NodeLst"))
			{
			chunk.initial = jb_NumNodes * 16;
			chunk.dataLength = chunk.initial;
			chunk.dataBuffer = (short *)malloc(chunk.dataLength);
			baseBuffer = (byte *)chunk.dataBuffer;
			chunk.version = 0;
			for (i = 0; i < jb_NumNodes; i++)
				{
				// 2 - Node name, offset in string bank
				*((unsigned short *)(baseBuffer)) = fs_stringBankAppend(strBuffer, strLength, jb_Node[i].type->name);
				baseBuffer += sizeof(unsigned short);
				// 2 - Item name, offset in string bank
				if (jb_Node[i].type->linksEntity)
					{
					for (j = 0; j < jb_NumItems; j++)
						{
						if (jb_ItemTable[j].node != i)
							continue;
						if (jb_ItemTable[j].ent == NULL)
							break;
						*((unsigned short *)(baseBuffer)) = fs_stringBankAppend(strBuffer, strLength, jb_ItemTable[j].ent->classname);
						break;
						}
					}
				else
					*((unsigned short *)(baseBuffer)) = 0;
				baseBuffer += sizeof(unsigned short);
				// 3x 4 - Coordinates
				*((float *)(baseBuffer)) = jb_Node[i].origin[0]; baseBuffer += sizeof(float);
				*((float *)(baseBuffer)) = jb_Node[i].origin[1]; baseBuffer += sizeof(float);
				*((float *)(baseBuffer)) = jb_Node[i].origin[2]; baseBuffer += sizeof(float);
				}
			}

		// DATA BUFFER // Jump links //
		else if (!strcmp(label, "LinkJmp"))
			{
			chunk.initial = jb_NumJumps * 4;
			chunk.dataLength = chunk.initial;
			chunk.dataBuffer = (short *)malloc(chunk.dataLength);
			baseBuffer = (byte *)chunk.dataBuffer;
			chunk.version = 0;
			for (i = 0; i < jb_NumJumps; i++)
				{
				*((short *)(baseBuffer)) = jb_JumpTable[i].from;	baseBuffer += sizeof(short);	// 2 - First node
				*((short *)(baseBuffer)) = jb_JumpTable[i].to;		baseBuffer += sizeof(short);	// 2 - Target node
				}
			}

		// DATA BUFFER // Duck Links //
		else if (!strcmp(label, "LinkDuk"))
			{
			chunk.initial = jb_NumDucks * 4;
			chunk.dataLength = chunk.initial;
			chunk.dataBuffer = (short *)malloc(chunk.dataLength);
			chunk.version = 0;
			baseBuffer = (byte *)chunk.dataBuffer;
			for (i = 0; i < jb_NumDucks; i++)
				{
				*((short *)(baseBuffer)) = jb_DuckTable[i].from;	baseBuffer += sizeof(short);	// 2 - First node
				*((short *)(baseBuffer)) = jb_DuckTable[i].to;		baseBuffer += sizeof(short);	// 2 - Target node
				}
			}


		// DATA BUFFER // User Links //
		else if (!strcmp(label, "LinkUsr"))
			{
			chunk.initial = jb_NumLinks * 6;
			chunk.dataLength = chunk.initial;
			chunk.dataBuffer = (short *)malloc(chunk.dataLength);
			chunk.version = 0;
			baseBuffer = (byte *)chunk.dataBuffer;
			for (i = 0; i < jb_NumLinks; i++)
				{
				*((short *)(baseBuffer)) = jb_LinkTable[i].from;		baseBuffer += sizeof(short);	// 2 - First node
				*((short *)(baseBuffer)) = jb_LinkTable[i].to;			baseBuffer += sizeof(short);	// 2 - Target node
				// 2 - Move type
				if (jb_LinkTable[i].vehicle == VEHICLE_WALK)
					*((unsigned short *)(baseBuffer)) = fs_stringBankAppend(strBuffer, strLength, "walk");
				else if (jb_LinkTable[i].vehicle == VEHICLE_JUMP)
					*((unsigned short *)(baseBuffer)) = fs_stringBankAppend(strBuffer, strLength, "jump");
				else if (jb_LinkTable[i].vehicle == VEHICLE_CROUCH)
					*((unsigned short *)(baseBuffer)) = fs_stringBankAppend(strBuffer, strLength, "duck");
				else
					*((unsigned short *)(baseBuffer)) = fs_stringBankAppend(strBuffer, strLength, "null");
				baseBuffer += sizeof(short);
				}
			}

		else
			{
			gi.dprintf("Chunk label %s unknown, check your code you dummy!\n", label);
			return chunk;
			}

		// CHUNK DEFINITION //
		chunk.label = fs_stringBankAppend(strBuffer, strLength, label);

		// SHRINK IF POSSIBLE - NEVER IN QUICKSAVE MODE //
		if (!quicksave)
			{
			if ((int)bot_nodeshrink->value & 1)
				rle_Shrink(&chunk.dataBuffer, &chunk.dataLength, &chunk.rleBuffer, &chunk.rleLength);
			if ((int)bot_nodeshrink->value & 2)
				wr_Shrink(&chunk.dataBuffer, &chunk.dataLength, &chunk.wrBuffer, &chunk.wrLength);
			}

		// COMPUTE PADDING //
		i = (chunk.dataLength + chunk.rleLength + chunk.wrLength);
		if (i % 16)
			chunk.padding = (16 - (i % 16));
		else
			chunk.padding = 0;

		return chunk;
		}


	
	/***************************************************************************

	  Get string offset from the bank (super safe: read only, never changes
		pointer or length of string bank)

	***************************************************************************/
	unsigned short int fs_stringBankOffset(byte *strBuffer, unsigned int strLength, char *msg)
		{
		int		ofs = 0;
		int   cnt = 0;
		char  *strPtr;

		// No message to copy
		if (!(msg && strLength))
			return 0;

		strPtr = ((char *)strBuffer);
		do
			{
			if (!Q_stricmp(&strPtr[ofs], msg))
				return ofs;
			ofs += (strlen(&strPtr[ofs]) + 1);
			} while (ofs < strLength);

		return 0;
		}



	/********************************************************************
	
		Append characters in a byte buffer, resize buffer as needed.
		REMEMBER TO FREE MEMORY <strBuffer> WHEN YOU'RE DONE!

	********************************************************************/
	unsigned short int fs_stringBankAppend(byte **strBuffer, unsigned short int *strLength, char *msg)
		{
		int									ofs = 0;
		unsigned short int	msgLength = strlen(msg) + 1;
		unsigned short int	oldLength = *strLength;
		unsigned short int	newLength = oldLength + msgLength;
		char								*strPtr;
		byte								*newBuffer;
		byte								*strPointer;

		// No message to copy
		if (!msg)
			return 0;

		// If <msg> is already in the bank, exit
		if (oldLength)
			{
			strPtr = ((char *)(*strBuffer));
			do
				{
				if (!Q_stricmp(&(strPtr[ofs]), msg))
					return ofs;
				ofs += (strlen(&(strPtr[ofs])) + 1);
				} while (ofs < *strLength);
			}

		// Reallocate memory so we can store this string
		newBuffer = (byte *)realloc(*strBuffer, newLength);
		if (!newBuffer)
			{
			gi.dprintf("fs_stringBankAppend: unable to reallocate memory.\n");
			return 0;
			}

		// Copy string at the end of new buffer
		strPointer = (byte *)(msg);
		for (ofs = 0; ofs < msgLength; ofs++)
			newBuffer[(*strLength) + ofs] = strPointer[ofs];

		// Done
		*strBuffer = newBuffer;
		*strLength = newLength;
		return oldLength;
		}
		


	/********************************************************************

		Free string bank

	********************************************************************/
	void fs_stringBankDestroy(byte **strBuffer, unsigned short int *strLength)
		{
		if (!(*strBuffer))
			return;
		free(*strBuffer);
		*strBuffer = 0;
		*strLength = 0;
		}



	/********************************************************************
	
		Setup string bank: create a NULL entry, insert node & entity names.
		<strBuffer> must be available (unused)

	********************************************************************/
	void fs_stringBankSetup(byte **strBuffer, unsigned short int *strLength, qboolean start)
		{
		int i;

		// Setup NULL string
		(*strBuffer) = (byte *)malloc(1);
		if ((*strBuffer) == NULL)
			{
			gi.dprintf("fs_stringBankSetup: unable to allocate memory.\n");
			return;
			}
		(*strBuffer)[0]	= 0x00;
		(*strLength) 		= 1;

		if (start)
			{
			// Add node types
			for (i = 0; i < jb_NumNodes; i++)
				{
				if (!jb_Node[i].type)
					continue;
				fs_stringBankAppend(strBuffer, strLength, jb_Node[i].type->name);
				}

			// Add entity classnames
			for (i = 0; i < jb_NumItems; i++)
				{
				if (jb_ItemTable[i].ent == NULL)
					continue;
				if (jb_ItemTable[i].ent->classname)
					fs_stringBankAppend(strBuffer, strLength, jb_ItemTable[i].ent->classname);
				}
			}
		}
#endif